<?php

declare(strict_types=1);

namespace Drupal\pinto_theme\Render;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\RefinableCacheableDependencyInterface;
use Drupal\Core\Render\AttachmentsInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\Render\RendererInterface;
use Drupal\pinto_theme\PintoTheme\Html\PintoThemeHtmlContext;
use Pinto\PintoMapping;

/**
 * Decorator for Renderer.
 *
 * Overrides rendering of #type=html as commanded by HtmlRenderer.
 */
final class PintoThemeRenderer implements RendererInterface {

  public const PINTO_THEME_CONTEXT = '#pinto-theme-html';

  public function __construct(
    private RendererInterface $inner,
    private PintoMapping $pintoMapping,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function renderRoot(&$elements) {
    return $this->inner->renderRoot($elements);
  }

  /**
   * {@inheritdoc}
   */
  public function renderInIsolation(&$elements) {
    return $this->inner->renderInIsolation($elements);
  }

  /**
   * {@inheritdoc}
   */
  public function renderPlain(&$elements) {
    return $this->inner->renderPlain($elements);
  }

  /**
   * {@inheritdoc}
   */
  public function renderPlaceholder($placeholder, array $elements) {
    return $this->inner->renderPlaceholder($placeholder, $elements);
  }

  /**
   * {@inheritdoc}
   */
  public function render(&$elements, $is_root_call = FALSE) {
    if (isset($elements['#type']) && $elements['#type'] === 'html' && isset($elements['page']) && \array_key_exists(static::PINTO_THEME_CONTEXT, $elements['page'])) {
      // As provided by
      // \Drupal\pinto_theme\PintoTheme\PintoThemeInterface::htmlObject.
      // via
      // \Drupal\pinto_theme\Plugin\DisplayVariant\PintoThemePageVariant::build().
      /** @var \Drupal\pinto_theme\Render\PintoRenderContext $context */
      $context = $elements['page'][static::PINTO_THEME_CONTEXT];
      $componentClass = $context->component;

      $htmlObject = $componentClass::createHtmlObjectFrom(PintoThemeHtmlContext::createHtmlContextFrom(
        $context->title,
        $context->content,
        $context->routeMatch,
        \array_key_exists('page_top', $elements),
        \array_key_exists('page_bottom', $elements),
        $elements['page_top'] ?? NULL,
        $elements['page_bottom'] ?? NULL,
      ));

      // Record #attached and #cache keys to be brought back in. If the Pinto
      // object classes implement appropriate interfaces, add them in there
      // instead.
      $pageAttached = $elements['page']['#attached'];
      $pageCache = $elements['page']['#cache'];
      // Return #attached and #cache as brought in by \Drupal\Core\Render\MainContent\HtmlRenderer::invokePageAttachmentHooks.
      if ($htmlObject instanceof AttachmentsInterface) {
        $htmlObject->addAttachments($pageAttached);
        $pageAttached = NULL;
      }
      if ($htmlObject instanceof RefinableCacheableDependencyInterface) {
        $htmlObject->addCacheableDependency(CacheableMetadata::createFromRenderArray(['#cache' => $pageCache]));
        $pageCache = NULL;
      }

      $builder = $this->pintoMapping->getBuilder($htmlObject);
      $build = $builder();
      if (FALSE === \is_array($build)) {
        throw new \LogicException('Not a render array.');
      }

      // Handle when the page didn't accept cache or attach directly.
      // Note that these are applied to HTML render array since we can't be sure
      // of where the page object was rendered out to within the HTML render
      // array.
      if ($pageCache !== NULL) {
        CacheableMetadata::createFromRenderArray($build)
          ->merge(CacheableMetadata::createFromRenderArray(['#cache' => $pageCache]))
          ->applyTo($build);
      }
      if ($pageAttached !== NULL) {
        BubbleableMetadata::createFromRenderArray($build)
          ->merge(BubbleableMetadata::createFromRenderArray(['#attached' => $pageAttached]))
          ->applyTo($build);
      }

      // @todo TEMPORARY!. Process in ObjectTrait::__invoke, instead of here.
      // since \Drupal\pinto_theme_demo\Pinto\Html adds attachments from within.
      if ($htmlObject instanceof AttachmentsInterface) {
        BubbleableMetadata::createFromRenderArray($build)
          ->merge(BubbleableMetadata::createFromObject($htmlObject))
          ->applyTo($build);
      }

      $elements = $build;
    }

    return $this->inner->render($elements, $is_root_call);
  }

  /**
   * {@inheritdoc}
   */
  public function hasRenderContext() {
    return $this->inner->hasRenderContext();
  }

  /**
   * {@inheritdoc}
   */
  public function executeInRenderContext($context, $callable) {
    return $this->inner->executeInRenderContext($context, $callable);
  }

  /**
   * {@inheritdoc}
   */
  public function mergeBubbleableMetadata($a, $b) {
    return $this->inner->mergeBubbleableMetadata($a, $b);
  }

  /**
   * {@inheritdoc}
   */
  public function addCacheableDependency(&$elements, $dependency) {
    return $this->inner->addCacheableDependency($elements, $dependency);
  }

}
