<?php

declare(strict_types=1);

namespace Drupal\pinto_theme_test\Pinto\Html;

use Drupal\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Render\AttachmentsInterface;
use Drupal\Core\Render\AttachmentsTrait;
use Drupal\Core\Render\Markup;
use Drupal\Core\Template\Attribute;
use Drupal\pinto\Object\DrupalInvokableSlotsTrait;
use Drupal\pinto_theme\Html\DrupalHtmlTemplate;
use Drupal\pinto_theme\PintoTheme\Attribute\Theme;
use Drupal\pinto_theme\PintoTheme\Html\PintoThemeHtmlContext;
use Drupal\pinto_theme\PintoTheme\Html\PintoThemeHtmlObjectInterface;
use Pinto\Attribute\Asset;
use Pinto\Attribute\ObjectType;
use Pinto\Slots;

#[
  ObjectType\Slots(bindPromotedProperties: TRUE),
  Theme(themeName: 'empty_theme'),
  Asset\Css,
]
final class Html implements AttachmentsInterface, PintoThemeHtmlObjectInterface {

  use DrupalInvokableSlotsTrait;
  use AttachmentsTrait;

  public function __construct(
    public mixed $content,
    public mixed $title,
    public mixed $top = NULL,
    public mixed $bottom = NULL,
    public string $placeholderToken = '',
    public string $htmlTitle = '',
    public Attribute $htmlAttributes = new Attribute(),
    public Attribute $attributes = new Attribute(),
  ) {
  }

  public static function createHtmlObjectFrom(PintoThemeHtmlContext $context): object {
    return new static(
      content: $context->getContent(),
      title: $context->getTitle(),
      top: $context->hasTop() ? $context->getTop() : NULL,
      bottom: $context->hasBottom() ? $context->getBottom() : NULL,
    );
  }

  protected function build(Slots\Build $build): Slots\Build {
    $vars = DrupalHtmlTemplate::createFromContainer(static::getContainer());
    $vars->applyAttachments($this);

    $siteName = static::configFactory()->get('system.site')->get('name');

    $build
      ->set('attributes', $this->attributes)
      ->set('htmlAttributes', $vars->htmlAttributes())
      // And strip TWIG debug (!)
      // @phpstan-ignore-next-line
      ->set('htmlTitle', Markup::create(\trim(\strip_tags(\sprintf('%s | %s', $vars->renderTitle($this->title), $siteName)))))
      ->set('top', $this->top)
      ->set('bottom', $this->bottom)
      ->set('placeholderToken', $vars->placeHolderToken());

    return $build;
  }

  private static function configFactory(): ConfigFactoryInterface {
    return \Drupal::configFactory();
  }

  private static function getContainer(): ContainerInterface {
    return \Drupal::getContainer();
  }

}
