<?php

declare(strict_types=1);

namespace Drupal\pinto_theme\EventListener;

use Drupal\Core\Render\PageDisplayVariantSelectionEvent;
use Drupal\Core\Render\RenderEvents;
use Drupal\Core\Theme\ThemeManagerInterface;
use Drupal\pinto_theme\Plugin\DisplayVariant\PintoThemePageVariant;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Informs Drupal that Pinto will render a page.
 *
 * - Instead of a page built with blocks.
 * - And, only when the active theme maps to a Pinto list.
 *
 * @see \Drupal\Core\Render\MainContent\HtmlRenderer::prepare
 * @phpstan-import-type PintoThemePageVariantConfiguration from \Drupal\pinto_theme\Plugin\DisplayVariant\PintoThemePageVariant
 */
final class PintoThemePageDisplayVariant implements EventSubscriberInterface {

  /**
   * Enums keyed by theme name.
   *
   * @var array<string, class-string<object>>
   */
  private array $mapping;

  public function __construct(
    private readonly ThemeManagerInterface $themeManager,
    #[Autowire(param: 'pinto_theme.components')]
    string $mapping,
  ) {
    // @phpstan-ignore-next-line
    $this->mapping = \unserialize($mapping);
  }

  public function onSelectPageDisplayVariant(PageDisplayVariantSelectionEvent $event): void {
    $activeThemeName = $this->themeManager->getActiveTheme($event->getRouteMatch())->getName();

    $componentClass = $this->mapping[$activeThemeName] ?? NULL;
    if (NULL === $componentClass) {
      // Quit if active theme is not associated with a Pinto component.
      return;
    }

    // See https://www.drupal.org/project/coder/issues/3523789.
    // phpcs:ignore Drupal.Commenting.DataTypeNamespace.DataTypeNamespace
    /** @var PintoThemePageVariantConfiguration $configuration */
    $configuration = [
      'pintoTheme' => $componentClass,
      'pintoThemeContext' => [$event->getRouteMatch()],
    ];

    $event
      ->setPluginId(PintoThemePageVariant::PLUGIN_ID)
      ->setPluginConfiguration($configuration);
  }

  public static function getSubscribedEvents(): array {
    $events = [];
    $events[RenderEvents::SELECT_PAGE_DISPLAY_VARIANT][] = ['onSelectPageDisplayVariant'];
    return $events;
  }

}
