<?php

namespace Drupal\piwik_pro_dashboard\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\piwik_pro_dashboard\Enum\DateRange;
use Drupal\piwik_pro_dashboard\Service\PiwikApiClient;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides API endpoints for the Piwik PRO Dashboard module.
 */
class ApiController extends ControllerBase {

  /**
   * The Piwik API client service.
   *
   * @var \Drupal\piwik_pro_dashboard\Service\PiwikApiClient
   */
  protected PiwikApiClient $apiClient;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $logger;

  /**
   * Constructs the ApiController.
   *
   * @param \Drupal\piwik_pro_dashboard\Service\PiwikApiClient $api_client
   *   The Piwik API client service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger
   *   The logger service.
   */
  public function __construct(PiwikApiClient $api_client, LoggerChannelFactoryInterface $logger) {
    $this->apiClient = $api_client;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   *
   * Creates an instance of the controller.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   *
   * @return static
   *   An instance of the ApiController.
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('piwik_pro_dashboard.api_client'),
      $container->get('logger.factory')
    );
  }

  /**
   * Handles the "overview" API endpoint.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response containing the overview data or an error message.
   */
  public function getOverview(Request $request): JsonResponse {
    try {
      $aggregated = $request->query->get('aggregated') === 'true';
      $range = $request->query->get('range');
      if ($range && !DateRange::tryFrom($range)) {
        throw new \InvalidArgumentException('Invalid date range provided.');
      }
      else {
        $range = DateRange::tryFrom($range) ?? DateRange::Last7Days;
      }
      $data = $this->apiClient->fetchOverviewData($aggregated, $range);
      return new JsonResponse($data);
    }
    catch (\Exception $e) {
      $this->logger->get('piwik_pro_dashboard')->error('API error: @message', ['@message' => $e->getMessage()]);
      return new JsonResponse([
        'error' => 'Unable to fetch data from Piwik PRO.',
      ], 500);
    }
  }

  /**
   * Handles the "devices" API endpoint.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response containing the device data or an error message.
   */
  public function getDevices(Request $request): JsonResponse {
    try {
      $range = $request->query->get('range');
      if ($range && !DateRange::tryFrom($range)) {
        throw new \InvalidArgumentException('Invalid date range provided.');
      }
      else {
        $range = DateRange::tryFrom($range) ?? DateRange::Last7Days;
      }
      $data = $this->apiClient->fetchDeviceData($range);
      return new JsonResponse($data);
    }
    catch (\Exception $e) {
      $this->logger->get('piwik_pro_dashboard')->error('API error: @message', ['@message' => $e->getMessage()]);
      return new JsonResponse([
        'error' => 'Unable to fetch data from Piwik PRO.',
      ], 500);
    }
  }

  /**
   * Handles the "top pages" API endpoint.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response containing the top pages data or an error message.
   */
  public function getTopPages(Request $request): JsonResponse {
    try {
      $range = $request->query->get('range');
      if ($range && !DateRange::tryFrom($range)) {
        throw new \InvalidArgumentException('Invalid date range provided.');
      }
      else {
        $range = DateRange::tryFrom($range) ?? DateRange::Last7Days;
      }
      $data = $this->apiClient->fetchToPages($range);
      return new JsonResponse($data);
    }
    catch (\Exception $e) {
      $this->logger->get('piwik_pro_dashboard')->error('API error: @message', ['@message' => $e->getMessage()]);
      return new JsonResponse([
        'error' => 'Unable to fetch data from Piwik PRO.',
      ], 500);
    }
  }

}
