<?php

namespace Drupal\piwik_pro_dashboard\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\piwik_pro_dashboard\Helper\ClientNameHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Provides a controller for rendering the Piwik PRO Dashboard.
 */
class DashboardController extends ControllerBase {

  /**
   * The client name helper service.
   *
   * @var \Drupal\piwik_pro_dashboard\Helper\ClientNameHelper
   */
  protected $clientNameHelper;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $logger;

  /**
   * Constructs the DashboardController.
   *
   * @param \Drupal\piwik_pro_dashboard\Helper\ClientNameHelper $clientNameHelper
   *   The client name helper service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger
   *   The logger service.
   */
  public function __construct(ClientNameHelper $clientNameHelper, LoggerChannelFactoryInterface $logger) {
    $this->clientNameHelper = $clientNameHelper;
    $this->logger = $logger;
  }

  /**
   * Factory method for creating an instance of the DashboardController.
   *
   * This method is used by the dependency injection container to instantiate
   * the controller with the required services.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   *
   * @return static
   *   An instance of the DashboardController.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('piwik_pro_dashboard.client_name_helper'),
      $container->get('logger.factory')
    );
  }

  /**
   * Renders the Piwik PRO Dashboard application.
   *
   * @return array
   *   A render array containing the dashboard markup and attached libraries.
   */
  public function render(): array {
    $clientName = '';

    // Extract the client name from the Piwik container URL.
    try {
      $clientName = $this->clientNameHelper->getClientName();
      if (!$clientName) {
        $clientName = 'Unknown client, check main Piwik Pro config page.';
      }
    }
    catch (\Exception $e) {
      $this->logger->get('piwik_pro_dashboard')->error('Failed to extract client name: @message', ['@message' => $e->getMessage()]);
    }

    return [
      '#markup' => '<h2>' . $this->t('Piwik PRO account: @name', ['@name' => $clientName]) . '</h2><div id="piwik-dashboard-app"></div>',
      '#attached' => [
        'library' => [
          'piwik_pro_dashboard/dashboard_app',
        ],
      ],
    ];
  }

}
