<?php

namespace Drupal\piwik_pro_dashboard\Helper;

use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Helper class for extracting the client name (subdomain) from a Piwik PRO URL.
 */
class ClientNameHelper {

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs the ClientNameHelper service.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory service.
   */
  public function __construct(ConfigFactoryInterface $configFactory) {
    $this->configFactory = $configFactory;
  }

  /**
   * Extracts the client name (subdomain) from a Piwik PRO URL.
   *
   * @return string|null
   *   The extracted subdomain (e.g., 'yourname'), or NULL if invalid.
   */
  public function getClientName(): ?string {
    $piwikConfig = $this->configFactory->get('piwik_pro.settings');
    $url = $piwikConfig->get('piwik_domain');

    if (empty($url)) {
      throw new \InvalidArgumentException('Piwik domain URL is not set.');
    }

    // Add scheme if missing.
    if (stripos($url, 'http://') !== 0 && stripos($url, 'https://') !== 0) {
      $url = 'https://' . $url;
    }

    $host = parse_url($url, PHP_URL_HOST);
    if (!$host) {
      return NULL;
    }

    // Strip www.
    if (str_starts_with($host, 'www.')) {
      $host = substr($host, 4);
    }

    $parts = explode('.', $host);
    $count = count($parts);

    if ($count === 3 && $parts[1] === 'piwik' && $parts[2] === 'pro') {
      return $parts[0];
    }

    if ($count === 4 && $parts[1] === 'containers' && $parts[2] === 'piwik' && $parts[3] === 'pro') {
      return $parts[0];
    }

    return NULL;
  }

}
