<?php

namespace Drupal\piwik_pro_dashboard\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\key\Entity\Key;

/**
 * Provides a settings form for the Piwik PRO Dashboard module.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   *
   * Returns the unique ID of the settings form.
   *
   * @return string
   *   The form ID.
   */
  public function getFormId(): string {
    return 'piwik_pro_dashboard_settings';
  }

  /**
   * {@inheritdoc}
   *
   * Returns the names of the editable configuration objects.
   *
   * @return array
   *   An array of configuration names.
   */
  protected function getEditableConfigNames(): array {
    return ['piwik_pro_dashboard.settings'];
  }

  /**
   * {@inheritdoc}
   *
   * Builds the settings form.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The form structure.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('piwik_pro_dashboard.settings');

    // List of available keys from the Key module.
    $key_options = [];
    foreach (Key::loadMultiple() as $key) {
      $key_options[$key->id()] = $key->label();
    }

    $form['#prefix'] = $this->t('
      <div>For the dashboard to be functional, you need to obtain client credentials from Piwik PRO.
      More info here: <a target="_blank" href="https://developers.piwik.pro/reference/authentication">
      https://developers.piwik.pro/reference/authentication</a>.
      </div><div>You can view the dashboard <a href="/admin/reports/piwik-dashboard">here</a>.</div>
    ');

    $form['client_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Client ID'),
      '#default_value' => $config->get('client_id'),
      '#required' => TRUE,
    ];

    $form['client_secret'] = [
      '#type' => 'select',
      '#title' => $this->t('Client secret (stored in Key module)'),
      '#options' => $key_options,
      '#default_value' => $config->get('client_secret'),
      '#required' => TRUE,
      '#description' => $this->t('Select a key containing your Piwik PRO client secret.
      WARNING: we recommend against storing the API key in your configuration files.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   *
   * Handles form submission and saves the configuration.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('piwik_pro_dashboard.settings')
      ->set('client_id', $form_state->getValue('client_id'))
      ->set('client_secret', $form_state->getValue('client_secret'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
