<?php

namespace Drupal\piwik_pro_dashboard\Service;

use GuzzleHttp\ClientInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\piwik_pro_dashboard\Helper\ClientNameHelper;

/**
 * Manages access tokens for the Piwik PRO API.
 */
class AccessTokenManager {

  /**
   * HTTP client for making API requests.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * Cache backend for storing the access token.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cache;

  /**
   * Logger for logging errors and messages.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $logger;

  /**
   * Configuration factory for accessing module settings.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Entity type manager for loading entities.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Configuration for the Piwik PRO Dashboard module.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $piwikdashboardConfig;

  /**
   * Helper for extracting the client name from the Piwik domain URL.
   *
   * @var \Drupal\piwik_pro_dashboard\Helper\ClientNameHelper
   */
  protected $clientNameHelper;
  /**
   * The client name extracted from the Piwik domain URL.
   *
   * @var string
   */
  protected $clientName;

  /**
   * Constructs the AccessTokenManager service.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client for making API requests.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend for storing the access token.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $factory
   *   The logger factory for logging messages.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory for accessing settings.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager for loading entities.
   * @param \Drupal\piwik_pro_dashboard\Helper\ClientNameHelper $clientNameHelper
   *   The client name helper for extracting the client name
   *   from the Piwik Pro domain URL.
   *
   * @throws \InvalidArgumentException
   *   Thrown when the Piwik domain URL is invalid or not set.
   */
  public function __construct(ClientInterface $http_client, CacheBackendInterface $cache, LoggerChannelFactoryInterface $factory, ConfigFactoryInterface $configFactory, EntityTypeManagerInterface $entityTypeManager, ClientNameHelper $clientNameHelper) {
    $this->httpClient = $http_client;
    $this->cache = $cache;
    $this->logger = $factory;
    $this->configFactory = $configFactory;
    $this->entityTypeManager = $entityTypeManager;
    $this->clientNameHelper = $clientNameHelper;

    $this->piwikdashboardConfig = $configFactory->get('piwik_pro_dashboard.settings');

    // Extract the client name from the Piwik container URL.
    try {
      $clientName = $this->clientNameHelper->getClientName();
      if ($clientName) {
        $this->clientName = $clientName;
      }
      else {
        throw new \InvalidArgumentException('Invalid Piwik domain URL.');
      }
    }

    catch (\Exception $e) {
      $this->logger->get('piwik_pro_dashboard')->error('Failed to extract client name: @message', ['@message' => $e->getMessage()]);
      throw new \RuntimeException('Unable to extract client name from the Piwik domain URL.');
    }
  }

  /**
   * Retrieves the access token for the Piwik PRO API.
   *
   * @return string
   *   The access token.
   *
   * @throws \RuntimeException
   *   Thrown when the access token cannot be retrieved.
   */
  public function getAccessToken(): string {
    if ($cached = $this->cache->get('piwik_pro_dashboard.access_token')) {
      return $cached->data;
    }

    $clientName = $this->clientName;
    $clientId = $this->piwikdashboardConfig->get('client_id');
    $keyId = $this->piwikdashboardConfig->get('client_secret');

    $apiKey = NULL;
    if ($keyId) {
      $key = $this->entityTypeManager->getStorage('key')->load($keyId);
      if ($key) {
        $apiKey = $key->getKeyValue();
      }
    }

    $tokenUrl = "https://{$clientName}.piwik.pro/auth/token";

    try {
      $response = $this->httpClient->request('POST', $tokenUrl, [
        'headers' => [
          'Content-Type' => 'application/json',
        ],
        'body' => json_encode([
          'grant_type' => 'client_credentials',
          'client_id' => $clientId,
          'client_secret' => $apiKey,
        ]),
      ]);

      $data = json_decode($response->getBody(), TRUE);
      $accessToken = $data['access_token'];
      $expiresIn = $data['expires_in'] ?? 1800;

      // Cache the token slightly below actual expiry time
      // to avoid issues with caching an expired token.
      $this->cache->set('piwik_pro_dashboard.access_token', $accessToken, time() + $expiresIn - 60);

      return $accessToken;

    }
    catch (\Exception $e) {
      $this->logger->get('piwik_pro_dashboard')->error('Failed to fetch Piwik PRO access token: @message', ['@message' => $e->getMessage()]);
      throw new \RuntimeException('Unable to get access token from Piwik PRO API.');
    }
  }

}
