<?php

namespace Drupal\piwik_pro_dashboard\Service;

use GuzzleHttp\ClientInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\piwik_pro_dashboard\Enum\DateRange;
use Drupal\piwik_pro_dashboard\Service\AccessTokenManager;
use Drupal\piwik_pro_dashboard\Helper\ClientNameHelper;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service for interacting with the Piwik PRO API.
 */
class PiwikApiClient {

  /**
   * HTTP client for making API requests.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * Manages access tokens for API authentication.
   *
   * @var \Drupal\piwik_pro_dashboard\Service\AccessTokenManager
   */
  protected AccessTokenManager $tokenManager;

  /**
   * Configuration for the Piwik PRO Dashboard module.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $piwikdashboardConfig;

  /**
   * Configuration for the Piwik PRO module.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $piwikConfig;

  /**
   * The client name extracted from the Piwik domain URL.
   *
   * @var string
   */
  protected $clientName;

  /**
   * The website ID for the Piwik PRO instance.
   *
   * @var string
   */
  protected $websiteId;

  /**
   * Helper for extracting the client name from the Piwik domain URL.
   *
   * @var \Drupal\piwik_pro_dashboard\Helper\ClientNameHelper
   */
  protected $clientNameHelper;

  /**
   * Logger for logging errors and messages.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $logger;

  /**
   * Constructs the PiwikApiClient service.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client for making API requests.
   * @param \Drupal\piwik_pro_dashboard\Service\AccessTokenManager $token_manager
   *   The access token manager for API authentication.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory for accessing settings.
   * @param \Drupal\piwik_pro_dashboard\Helper\ClientNameHelper $clientNameHelper
   *   The client name helper for extracting the client
   *   name from the Piwik domain URL.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $factory
   *   The logger factory for logging messages.
   *
   * @throws \InvalidArgumentException
   *   Thrown when the Piwik domain URL is invalid or not set.
   */
  public function __construct(ClientInterface $http_client, AccessTokenManager $token_manager, ConfigFactoryInterface $configFactory, ClientNameHelper $clientNameHelper, LoggerChannelFactoryInterface $factory) {
    $this->httpClient = $http_client;
    $this->tokenManager = $token_manager;
    $this->piwikdashboardConfig = $configFactory->get('piwik_pro_dashboard.settings');
    $this->piwikConfig = $configFactory->get('piwik_pro.settings');
    $this->clientNameHelper = $clientNameHelper;
    $this->websiteId = $this->piwikConfig->get('site_id');
    $this->logger = $factory;
    // Extract the client name from the Piwik container URL.
    try {
      $clientName = $this->clientNameHelper->getClientName();
      if ($clientName) {
        $this->clientName = $clientName;
      }
      else {
        throw new \InvalidArgumentException('Invalid Piwik domain URL.');
      }

    }
    catch (\Exception $e) {
      $this->logger->get('piwik_pro_dashboard')->error('Failed to extract client name: @message', ['@message' => $e->getMessage()]);
      throw new \RuntimeException('Unable to extract client name from the Piwik domain URL.');
    }
  }

  /**
   * Fetches overview data from the Piwik PRO API.
   *
   * @param bool $aggregated
   *   Whether to aggregate the data.
   * @param string $range
   *   The date range ('7d' or '30d').
   *
   * @return array
   *   The overview data.
   */
  public function fetchOverviewData(bool $aggregated = FALSE, DateRange $range = DateRange::Last7Days): array {
    $clientName = $this->clientName;
    $uri = "https://{$clientName}.piwik.pro/api/analytics/v1/query/";
    $today = date('Y-m-d');

    if ($range == DateRange::Last7Days) {
      $pastDate = date('Y-m-d', strtotime('-7 days'));
    }
    elseif ($range == DateRange::Last30Days) {
      $pastDate = date('Y-m-d', strtotime('-30 days'));
    }

    $columns = [];
    if (!$aggregated) {
      $columns[] = ['transformation_id' => 'to_date', 'column_id' => 'timestamp'];
    }

    $columns = array_merge($columns, [
      ['column_id' => 'visitors'],
      ['column_id' => 'page_views'],
      ['column_id' => 'returning_visitors_rate'],
      ['column_id' => 'bounce_rate'],
    ]);

    $body = [
      'date_from' => $pastDate,
      'date_to' => $today,
      'period' => 'range',
      'website_id' => $this->websiteId,
      'offset' => 0,
      'limit' => 30,
      'columns' => $columns,
      'order_by' => [
        [0, 'asc'],
      ],
      "format" => "json-kv",
      'filters' => NULL,
      'metric_filters' => NULL,
    ];

    $response = $this->httpClient->request('POST', $uri, [
      'headers' => [
        'Authorization' => 'Bearer ' . $this->tokenManager->getAccessToken(),
        'Content-Type' => 'application/json',
        'Accept' => 'application/json',
      ],
      'body' => json_encode($body),
    ]);

    return json_decode($response->getBody(), TRUE);
  }

  /**
   * Fetches device data from the Piwik PRO API.
   *
   * @param string $range
   *   The date range ('7d' or '30d').
   *
   * @return array
   *   The device data.
   */
  public function fetchDeviceData(DateRange $range = DateRange::Last7Days): array {
    $clientName = $this->clientName;
    $uri = "https://{$clientName}.piwik.pro/api/analytics/v1/query/";
    $today = date('Y-m-d');

    if ($range == DateRange::Last7Days) {
      $pastDate = date('Y-m-d', strtotime('-7 days'));
    }
    elseif ($range == DateRange::Last30Days) {
      $pastDate = date('Y-m-d', strtotime('-30 days'));
    }

    $columns = [
      ['column_id' => 'device_type'],
      ['column_id' => 'sessions'],
    ];

    $body = [
      'date_from' => $pastDate,
      'date_to' => $today,
      'website_id' => $this->websiteId,
      'offset' => 0,
      'limit' => 10,
      'columns' => $columns,
      "format" => "json-kv",
    ];

    $response = $this->httpClient->request('POST', $uri, [
      'headers' => [
        'Authorization' => 'Bearer ' . $this->tokenManager->getAccessToken(),
        'Content-Type' => 'application/json',
        'Accept' => 'application/json',
      ],
      'body' => json_encode($body),
    ]);

    return json_decode($response->getBody(), TRUE);
  }

  /**
   * Fetches top pages data from the Piwik PRO API.
   *
   * @param string $range
   *   The date range ('7d' or '30d').
   *
   * @return array
   *   The top pages data.
   */
  public function fetchToPages(DateRange $range = DateRange::Last7Days): array {
    $clientName = $this->clientName;
    $uri = "https://{$clientName}.piwik.pro/api/analytics/v1/query/";
    $today = date('Y-m-d');

    if ($range == DateRange::Last7Days) {
      $pastDate = date('Y-m-d', strtotime('-7 days'));
    }
    elseif ($range == DateRange::Last30Days) {
      $pastDate = date('Y-m-d', strtotime('-30 days'));
    }

    $columns = [
      ['column_id' => 'event_url'],
      ['column_id' => 'page_views'],
    ];

    $body = [
      'date_from' => $pastDate,
      'date_to' => $today,
      'website_id' => $this->websiteId,
      'offset' => 0,
      'limit' => 10,
      'columns' => $columns,
      "format" => "json-kv",
    ];

    $response = $this->httpClient->request('POST', $uri, [
      'headers' => [
        'Authorization' => 'Bearer ' . $this->tokenManager->getAccessToken(),
        'Content-Type' => 'application/json',
        'Accept' => 'application/json',
      ],
      'body' => json_encode($body),
    ]);

    return json_decode($response->getBody(), TRUE);
  }

}
