<?php

namespace Drupal\plotly\Plugin\PlotlyChart;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\plotly\Attribute\PlotlyChart;
use Drupal\plotly\PlotlyChartInterface;

/**
 * Plugin implementation of the plotly chart.
 */
#[PlotlyChart(
  id: 'bar',
  label: new TranslatableMarkup('Bar charts'),
  description: new TranslatableMarkup('Bar charts is used when you have many fields, each field is a type of display column.'),
  example: 'https://plotly.com/javascript/bar-charts/',
  category: 'Basic Charts',
)]
class Bar implements PlotlyChartInterface {

  /**
   * {@inheritDoc}
   */
  public function data(array $rows, array $options = [], $view = NULL, $layout = []): array {
    $x = $series = $texts = [];
    foreach ($rows as $row) {
      $render = $view->rowPlugin->render($row);
      $row_index = $render['#row']->index;
      $x[] = $view->style_plugin->getField($row_index, $options['label']);
      foreach (array_keys($options['fields']) as $field_name) {
        if ($field_name === $options['label']) {
          continue;
        }
        $value = $view->style_plugin->getField($row_index, $field_name);
        if ($field_name === $options['text']) {
          $texts[$field_name][] = $value;
          continue;
        }
        $series[$field_name][] = $value;
      }
    }
    $traces = [];
    foreach ($series as $field_name => $values) {
      $label = $options['fields'][$field_name]['name'] ?? $field_name;
      $color = $options['fields'][$field_name]['color'] ?? '';
      $display_type = $options['fields'][$field_name]['display_type'] ?? 'bar';
      $traces[] = [
        'name' => $label,
        'x' => !empty($options['layout']['config_charts']['horizontal']) ? $values : $x,
        'y' => !empty($options['layout']['config_charts']['horizontal']) ? $x : $values,
        'orientation' => !empty($options['layout']['config_charts']['horizontal']) ? 'h' : '',
        'text' => !empty($texts) ? current($texts) : '',
        'type' => $display_type,
        'marker' => [
          'color' => $color === '#000000' ? '' : $color,
        ],
      ];
    }
    return [
      'data' => $traces,
      'layout' => $layout + [
        'title' => [
          'text' => $options['title'] ?? new TranslatableMarkup('Bar Chart'),
        ],
        'annotations' => [[
          'text' => $options['description'],
          'showarrow' => FALSE,
          'xref' => 'paper',
          'yref' => 'paper',
          'y' => -0.1,
          'xanchor' => 'center',
          'yanchor' => 'top',
        ],
        ],
        'barmode' => !empty($options['stacked']) ? 'stack' : 'group',
      ],
    ];
  }

}
