<?php

namespace Drupal\plotly\Plugin\PlotlyChart;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\plotly\Attribute\PlotlyChart;
use Drupal\plotly\PlotlyChartInterface;

/**
 * Plugin implementation of the plotly chart.
 */
#[PlotlyChart(
  id: 'box',
  label: new TranslatableMarkup('Box Plots'),
  description: new TranslatableMarkup('Box Plots must have 7 values [min, lower, q1, median, q3, upper, max]'),
  example: 'https://plotly.com/javascript/box-plots/',
  category: 'Statistical Charts',
)]
class Box implements PlotlyChartInterface {

  /**
   * {@inheritDoc}
   */
  public function data(array $rows, array $options = [], $view = NULL, $layout = []): array {
    $label_values = $series = $texts = [];
    $orientation = !empty($options['horizontal']) ? 'x' : 'y';
    $label_name = empty($options['horizontal']) ? 'x' : 'y';
    foreach ($rows as $row) {
      $render = $view->rowPlugin->render($row);
      $row_index = $render['#row']->index;
      $label_value = $view->style_plugin->getField($row_index, $options['label']);
      $label_values[] = $label_value;
      foreach (array_keys($options['fields']) as $field_name) {
        if ($field_name === $options['label']) {
          continue;
        }
        $value = $view->style_plugin->getField($row_index, $field_name);
        $series[$field_name][] = $value;
      }
    }
    $traces = [];
    $display_type = 'box';
    foreach ($series as $field_name => $values) {
      $label = $options['fields'][$field_name]['name'] ?? $field_name;
      $color = $options['fields'][$field_name]['color'] ?? '';
      $traces[] = [
        'name' => $label,
        $orientation => $values,
        $label_name => $label_values,
        'type' => $display_type,
        'marker' => [
          'color' => $color === '#000000' ? '' : $color,
        ],
      ];
    }
    return [
      'data' => $traces,
      'layout' => $layout + [
        'title' => [
          'text' => $options['title'] ?? new TranslatableMarkup('Box Chart'),
        ],
        'annotations' => [
          [
            'text' => $options['description'],
            'showarrow' => FALSE,
            'xref' => 'paper',
            'yref' => 'paper',
            'y' => -0.1,
            'xanchor' => 'center',
            'yanchor' => 'top',
          ],
        ],
        'barmode' => !empty($options['stacked']) ? 'stack' : 'group',
      ],
    ];
  }

}
