<?php

namespace Drupal\plotly\Plugin\PlotlyChart;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\plotly\Attribute\PlotlyChart;
use Drupal\plotly\PlotlyChartInterface;

/**
 * Plugin implementation of the plotly chart.
 */
#[PlotlyChart(
  id: 'dot',
  label: new TranslatableMarkup('Dot Plots'),
  description: new TranslatableMarkup('Dot Plots'),
  example: 'https://plotly.com/javascript/dot-plots/',
  category: 'Basic Charts',
)]
class Dot implements PlotlyChartInterface {

  /**
   * {@inheritDoc}
   */
  public function data(array $rows, array $options = [], $view = NULL, $layout = []): array {
    $display_type = $options['webgl'] ? 'scattergl' : 'scatter';
    $y = $series = $texts = [];
    foreach ($rows as $row) {
      $render = $view->rowPlugin->render($row);
      $row_index = $render['#row']->index;
      $y[] = $view->style_plugin->getField($row_index, $options['label']);
      foreach (array_keys($options['fields']) as $field_name) {
        if ($field_name === $options['label']) {
          continue;
        }
        $value = $view->style_plugin->getField($row_index, $field_name);
        if ($field_name === $options['text']) {
          $texts[$field_name][] = $value;
          continue;
        }
        $series[$field_name][] = $value;
      }
    }
    $traces = [];
    $mode = 'markers';
    foreach ($series as $field_name => $values) {
      $label = $options['fields'][$field_name]['name'] ?? $field_name;
      $color = $options['fields'][$field_name]['color'] ?? '';
      $traces[] = [
        'name' => $label,
        'x' => $values,
        'y' => $y,
        'orientation' => !empty($options['horizontal']) ? 'h' : '',
        'text' => !empty($texts) ? current($texts) : '',
        'type' => $display_type,
        'mode' => $mode,
        'automargin' => TRUE,
        'marker' => [
          'color' => $color === '#000000' ? '' : $color,
          'size' => $options['fields'][$field_name]['size'] ?? 16,
          'symbol' => 'circle',
        ],
      ];
    }

    return [
      'data' => $traces,
      'layout' => $layout + [
        'title' => ['text' => $options['title'] ?? new TranslatableMarkup('Dot Plot')],
        'xaxis' => [
          'showgrid' => FALSE,
          'showline' => TRUE,
        ],
        'annotations' => [
          [
            'text' => $options['description'],
            'showarrow' => FALSE,
            'xref' => 'paper',
            'yref' => 'paper',
            'y' => -0.1,
            'xanchor' => 'center',
            'yanchor' => 'top',
          ],
        ],
        'hovermode' => 'closest',
      ],
    ];
  }

}
