<?php

namespace Drupal\plotly\Plugin\PlotlyChart;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\plotly\Attribute\PlotlyChart;
use Drupal\plotly\PlotlyChartInterface;

/**
 * Plugin implementation of the plotly chart.
 */
#[PlotlyChart(
  id: 'histogram2d',
  label: new TranslatableMarkup('2D Histograms'),
  description: new TranslatableMarkup('2D Histograms'),
  example: 'https://plotly.com/javascript/2D-Histogram/',
  category: 'Statistical Charts',
)]
class Histogram2D implements PlotlyChartInterface {

  /**
   * {@inheritDoc}
   */
  public function data(array $rows, array $options = [], $view = NULL, $layout = []): array {
    $series = $texts = [];
    $orientation = !empty($options['horizontal']) ? 'y' : 'x';
    foreach ($rows as $row) {
      $render = $view->rowPlugin->render($row);
      $row_index = $render['#row']->index;
      foreach (array_keys($options['fields']) as $field_name) {
        if ($field_name === $options['label']) {
          continue;
        }
        $value = $view->style_plugin->getField($row_index, $field_name);
        if ($field_name === $options['text']) {
          $texts[$field_name][] = $value;
          continue;
        }
        $series[$field_name][] = $value;
      }
    }
    $traces = [];
    $display_type = 'histogram2d';
    foreach ($series as $field_name => $values) {
      $label = $options['fields'][$field_name]['name'] ?? $field_name;
      $color = $options['fields'][$field_name]['color'] ?? '';
      $traces[] = [
        'name' => $label,
        $orientation => $values,
        'text' => !empty($texts) ? current($texts) : '',
        'opacity' => 0.7,
        'type' => $display_type,
        'marker' => [
          'color' => $color === '#000000' ? '' : $color,
        ],
      ];
    }
    return [
      'data' => $traces,
      'layout' => $layout + [
        'title' => [
          'text' => $options['title'] ?? new TranslatableMarkup('Histograms Chart'),
        ],
        'annotations' => [
          [
            'text' => $options['description'],
            'showarrow' => FALSE,
            'xref' => 'paper',
            'yref' => 'paper',
            'y' => -0.1,
            'xanchor' => 'center',
            'yanchor' => 'top',
          ],
        ],
        'barmode' => !empty($options['stacked']) ? 'stack' : 'overlay',
      ],
    ];
  }

}
