<?php

namespace Drupal\plotly\Plugin\PlotlyChart;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\plotly\Attribute\PlotlyChart;
use Drupal\plotly\PlotlyChartInterface;

/**
 * Plugin implementation of the plotly chart.
 */
#[PlotlyChart(
  id: 'linesgroup',
  label: new TranslatableMarkup('Lines Group Chart'),
  description: new TranslatableMarkup('Lines Chart is used when you have only 1 field value, which needs to be grouped by type'),
  example: 'https://plotly.com/javascript/bar-charts/',
  category: 'Basic Charts',
)]
class LinesGroup implements PlotlyChartInterface {

  /**
   * {@inheritDoc}
   */
  public function data(array $rows, array $options = [], $view = NULL, $layout = []): array {
    $traces = [];
    $data_by_group = [];

    foreach ($rows as $row) {
      $render = $view->rowPlugin->render($row);
      $row_index = $render['#row']->index;
      $label = (string) $view->style_plugin->getField($row_index, $options['label']);
      $group = !empty($options['group'])
        ? (string) $view->style_plugin->getField($row_index, $options['group'])
        : 'Default';

      foreach (array_keys($options['fields']) as $field_name) {
        if (in_array($field_name, [$options['label'], $options['group']])) {
          continue;
        }

        $value = strip_tags((string) $view->style_plugin->getField($row_index, $field_name));

        if (!isset($data_by_group[$group])) {
          $data_by_group[$group] = ['x' => [], 'y' => []];
        }

        $data_by_group[$group]['x'][] = $label;
        $data_by_group[$group]['y'][$field_name][] = $value;
        if ($field_name === $options['text']) {
         $data_by_group[$group]['text'][] = $value;
        }
      }
    }

    foreach ($data_by_group as $group_name => $data) {
      foreach ($data['y'] as $field_name => $values) {
        $field_config = $options['fields'][$field_name];
        $label = $field_config['name'] ?? $field_name;
        $color = $field_config['color'] ?? '';
        $display_type = $field_config['display_type'] ?? 'linesgroup';

        $traces[] = [
          'name' => $group_name,
          'x' => !empty($options['layout']['config_charts']['horizontal']) ? $values : $data['x'],
          'y' => !empty($options['layout']['config_charts']['horizontal']) ? $data['x'] : $values,
          'text' => $data['text'] ?? [],
          'type' => $display_type,
          'hoverinfo' => 'y',
          'orientation' => !empty($options['layout']['config_charts']['horizontal']) ? 'h' : 'v',
          'marker' => [
            'color' => $color === '#000000' ? '' : $color,
          ],
        ];
      }
    }

    $layout += [
      'title' => [
        'text' => $options['title'] ?? new TranslatableMarkup('Lines Group Chart'),
      ],
      'barmode' => !empty($options['layout']['config_charts']['stacked']) ? 'stack' : 'group',
      'annotations' => [[
        'text' => $options['description'] ?? '',
        'showarrow' => FALSE,
        'xref' => 'paper',
        'yref' => 'paper',
        'y' => -0.1,
        'xanchor' => 'center',
        'yanchor' => 'top',
      ]],
    ];

    return [
      'data' => $traces,
      'layout' => $layout
    ];
  }

}
