<?php

namespace Drupal\plotly\Plugin\PlotlyChart;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\plotly\Attribute\PlotlyChart;
use Drupal\plotly\PlotlyChartInterface;

/**
 * Plugin implementation of the plotly chart.
 */
#[PlotlyChart(
  id: 'ohlc',
  label: new TranslatableMarkup('Open High Low Close Charts'),
  description: new TranslatableMarkup('OHLC need date field, and 4 fields: open, high, low, close'),
  example: 'https://plotly.com/javascript/candlestick-charts/',
  category: 'Financial Charts',
)]
class Ohlc implements PlotlyChartInterface {

  /**
   * {@inheritDoc}
   */
  public function data(array $rows, array $options = [], $view = NULL, $layout = []): array {
    $display_type = 'ohlc';
    $x = $open = $high = $low = $close = $series = [];
    foreach ($rows as $row) {
      $render = $view->rowPlugin->render($row);
      $row_index = $render['#row']->index;
      $x[] = $view->style_plugin->getField($row_index, $options['label']);
      foreach (array_keys($options['fields']) as $field_name) {
        if ($field_name === $options['label']) {
          continue;
        }
        $value = $view->style_plugin->getField($row_index, $field_name);
        switch ($field_name) {
          case $options['text']:
            $open[] = $value;
            break;

          case $options['size']:
            $high[] = $value;
            break;

          case $options['parent']:
            $low[] = $value;
            break;

          case $options['close']:
            $close[] = $value;
            break;

          default:
            $series[$field_name][] = $value;
            break;
        }
      }
    }
    $data = [
      [
        'type' => $display_type,
        'x' => $x,
        'close' => $close,
        'high' => $high,
        'low' => $low,
        'open' => $open,
        'xaxis' => 'x',
        'yaxis' => 'y',
      ],
    ];

    foreach ($series as $field_name => $values) {
      $label = $options['fields'][$field_name]['name'] ?? $field_name;
      $color = $options['fields'][$field_name]['color'] ?? '';
      $mode = $options['fields'][$field_name]['mode'] ?? 'lines+markers';
      $type = $options['fields'][$field_name]['display_type'] ?? 'bar';

      $data[] = [
        'name' => $label,
        'x' => $x,
        'y' => !empty($options['horizontal']) ? $x : $values,
        'type' => $type,
        'mode' => $mode,
        'marker' => [
          'color' => $color === '#000000' ? '' : $color,
          'size' => $options['fields'][$field_name]['size'] ?? 10,
        ],
      ];
    }
    return [
      'data' => $data,
      'layout' => $layout + [
        'title' => [
          'text' => $options['title'] ?? new TranslatableMarkup('Scatter Plots'),
        ],
        'dragmode' => 'zoom',
        'annotations' => [
          [
            'text' => $options['description'],
            'xref' => 'paper',
            'yref' => 'paper',
            'y' => -0.1,
            'showarrow' => FALSE,
            'xanchor' => 'center',
            'yanchor' => 'top',
          ],
        ],
        'xaxis' => [
          'autorange' => TRUE,
          "rangeslider" => [
            "range" => [reset($x), end($x)],
          ],
          'type' => 'date',
        ],
        'yaxis' => [
          'autorange' => TRUE,
          'type' => 'linear',
        ],
      ],
    ];
  }

}
