<?php

namespace Drupal\plotly\Plugin\PlotlyChart;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\plotly\Attribute\PlotlyChart;
use Drupal\plotly\PlotlyChartInterface;

/**
 * Plugin implementation of the plotly chart.
 */
#[PlotlyChart(
  id: 'pie',
  label: new TranslatableMarkup('Pie charts'),
  description: new TranslatableMarkup('Pie charts'),
  example: 'https://plotly.com/javascript/pie-charts/',
  category: 'Basic Charts',
)]
class Pie implements PlotlyChartInterface {

  /**
   * {@inheritDoc}
   */
  public function data(array $rows, array $options = [], $view = NULL, $layout = []): array {
    $labels = $series = $texts = [];
    foreach ($rows as $row) {
      $render = $view->rowPlugin->render($row);
      $row_index = $render['#row']->index;
      $labels[] = $view->style_plugin->getField($row_index, $options['label']);
      foreach (array_keys($options['fields']) as $field_name) {
        if ($field_name === $options['label']) {
          continue;
        }
        $value = $view->style_plugin->getField($row_index, $field_name);
        if ($field_name === $options['text']) {
          $texts[$field_name][] = $value;
        }
        $series[$field_name][] = $value;
      }
    }
    $traces = [];
    $display_type = 'pie';
    $column = 0;
    foreach ($series as $field_name => $values) {
      $label = $options['fields'][$field_name]['name'] ?? $field_name;
      $color = $options['fields'][$field_name]['color'] ?? '';
      $size = $options['fields'][$field_name]['size'] ?? 0;
      $traces[] = [
        'name' => $label,
        'labels' => $labels,
        'values' => $values,
        'domain' => ['row' => 0, 'column' => $column++],
        'text' => !empty($texts) ? current($texts) : '',
        'hoverinfo' => $options['layout']['config_charts']['hoverinfo'],
        'automargin' => TRUE,
        'textposition' => 'inside',
        'insidetextorientation' => 'radial',
        'hole' => $size / 100,
        'type' => $display_type,
        'marker' => [
          'color' => $color === '#000000' ? '' : $color,
        ],
      ];
    }
    return [
      'data' => $traces,
      'layout' => $layout + [
        'title' => ['text' => $options['title'] ?? new TranslatableMarkup('Pie Chart')],
        'annotations' => [
          [
            'text' => $options['description'],
            'xref' => 'paper',
            'yref' => 'paper',
            'y' => -0.1,
            'showarrow' => FALSE,
            'xanchor' => 'center',
            'yanchor' => 'top',
          ],
        ],
        'hoverlabel' => [
          'showarrows' => FALSE,
        ],
        'hovermode' => 'closest',
        'grid' => ['rows' => 1, 'columns' => $column],
      ],
    ];
  }

}
