<?php

namespace Drupal\plotly\Plugin\PlotlyChart;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\plotly\Attribute\PlotlyChart;
use Drupal\plotly\PlotlyChartInterface;

/**
 * Plugin implementation of the plotly chart.
 */
#[PlotlyChart(
  id: 'time_series',
  label: new TranslatableMarkup('Time Series'),
  description: new TranslatableMarkup('Time Series'),
  example: 'https://plotly.com/javascript/time-series/',
  category: 'Financial Charts',
)]
class TimeSeries implements PlotlyChartInterface {

  /**
   * {@inheritDoc}
   */
  public function data(array $rows, array $options = [], $view = NULL, $layout = []): array {
    $display_type = $options['webgl'] ? 'scattergl' : 'scatter';
    $x = $series = [];
    foreach ($rows as $row) {
      $render = $view->rowPlugin->render($row);
      $row_index = $render['#row']->index;
      $x[] = $view->style_plugin->getField($row_index, $options['label']);
      foreach (array_keys($options['fields']) as $field_name) {
        if ($field_name === $options['label']) {
          continue;
        }
        $value = $view->style_plugin->getField($row_index, $field_name);
        $series[$field_name][] = $value;
      }
    }
    $traces = [];
    foreach ($series as $field_name => $values) {
      $label = $options['fields'][$field_name]['name'] ?? $field_name;
      $color = $options['fields'][$field_name]['color'] ?? '';
      $mode = 'lines';
      $traces[] = [
        'name' => $label,
        'x' => $x,
        'y' => $values,
        'type' => $display_type,
        'mode' => $mode,
        'line' => [
          'color' => $color === '#000000' ? '' : $color,
        ],
      ];
    }
    return [
      'data' => $traces,
      'layout' => $layout + [
        'title' => [
          'text' => $options['title'] ?? new TranslatableMarkup('Time Series'),
        ],
        'annotations' => [
          [
            'text' => $options['description'],
            'xref' => 'paper',
            'yref' => 'paper',
            'y' => -0.1,
            'showarrow' => FALSE,
            'xanchor' => 'center',
            'yanchor' => 'top',
          ],
        ],
        'xaxis' => [
          'autorange' => TRUE,
          'rangeselector' => [
            'buttons' => [
              [
                'count' => 1,
                'label' => '1m',
                'step' => 'month',
                'stepmode' => 'backward',
              ],
              [
                'count' => 6,
                'label' => '6m',
                'step' => 'month',
                'stepmode' => 'backward',
              ],
              [
                'step' => "all",
              ],
            ],
          ],
          "rangeslider" => [
            "range" => [reset($x), end($x)],
          ],
          'type' => 'date',
        ],
        'yaxis' => [
          'autorange' => TRUE,
          'type' => 'linear',
        ],
      ],
    ];
  }

}
