<?php

namespace Drupal\plotly\Plugin\PlotlyChart;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\plotly\Attribute\PlotlyChart;
use Drupal\plotly\PlotlyChartInterface;

/**
 * Plugin implementation of the plotly chart.
 */
#[PlotlyChart(
  id: 'tree_maps',
  label: new TranslatableMarkup('Treemaps'),
  description: new TranslatableMarkup('Treemaps charts'),
  example: 'https://plotly.com/javascript/treemaps/',
  category: 'Basic Charts',
)]
class Treemaps implements PlotlyChartInterface {

  /**
   * {@inheritDoc}
   */
  public function data(array $rows, array $options = [], $view = NULL, $layout = []): array {
    $labels = $series = $texts = $parents = [];
    foreach ($rows as $row) {
      $render = $view->rowPlugin->render($row);
      $row_index = $render['#row']->index;
      $labels[] = $view->style_plugin->getField($row_index, $options['label']);
      foreach (array_keys($options['fields']) as $field_name) {
        if ($field_name === $options['label']) {
          continue;
        }
        $value = $view->style_plugin->getField($row_index, $field_name);
        if ($field_name === $options['text']) {
          $texts[$field_name][] = $value;
          continue;
        }
        if ($field_name === $options['parent']) {
          $parents[$field_name][] = $value;
          continue;
        }
        $series[$field_name][] = $value;
      }
    }
    $traces = [];
    $display_type = 'treemap';
    foreach ($series as $field_name => $values) {
      $label = $options['fields'][$field_name]['name'] ?? $field_name;
      $traces[] = [
        'name' => $label,
        'type' => $display_type,
        'labels' => $labels,
        'values' => $values,
        'text' => !empty($texts) ? current($texts) : '',
        'parents' => !empty($parents) ? current($parents) : '',
        'outsidetextfont' => ['size' => 20, 'color' => "#377eb8"],
        'marker' => ['line' => ['width' => 2]],
        'pathbar' => ['visible' => FALSE],
      ];
    }

    return [
      'data' => $traces,
      'layout' => [
        'title' => ['text' => $options['title'] ?? new TranslatableMarkup('Tree maps')],
        'annotations' => [
          [
            'text' => $options['description'],
            'showarrow' => FALSE,
            'xref' => 'paper',
            'yref' => 'paper',
            'y' => -0.1,
            'xanchor' => 'center',
            'yanchor' => 'top',
          ],
        ],
      ],
    ];
  }

}
