<?php

namespace Drupal\plotly\Plugin\PlotlyChart;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\plotly\Attribute\PlotlyChart;
use Drupal\plotly\PlotlyChartInterface;

/**
 * Plugin implementation of the plotly chart.
 */
#[PlotlyChart(
  id: 'violin',
  label: new TranslatableMarkup('Violin Plots'),
  description: new TranslatableMarkup('Box Plots must have 7 values [min, lower, q1, median, q3, upper, max]'),
  example: 'https://plotly.com/javascript/violin/',
  category: 'Statistical Charts',
)]
class Violin implements PlotlyChartInterface {

  /**
   * {@inheritDoc}
   */
  public function data(array $rows, array $options = [], $view = NULL, $layout = []): array {
    $series = [];
    $orientation = !empty($options['horizontal']) ? 'x' : 'y';
    foreach ($rows as $row) {
      $render = $view->rowPlugin->render($row);
      $row_index = $render['#row']->index;
      foreach (array_keys($options['fields']) as $field_name) {
        if ($field_name === $options['label']) {
          continue;
        }
        $value = $view->style_plugin->getField($row_index, $field_name);
        $series[$field_name][] = $value;
      }
    }
    $traces = [];
    $display_type = 'violin';
    foreach ($series as $field_name => $values) {
      $label = $options['fields'][$field_name]['name'] ?? $field_name;
      $color = $options['fields'][$field_name]['color'] ?? '';
      $traces[] = [
        'name' => $label,
        $orientation => $values,
        'type' => $display_type,
        'points' => 'none',
        'box' => ['visible' => TRUE],
        'boxpoints' => FALSE,
        'opacity' => 0.6,
        'meanline' => ['visible' => TRUE],
        'marker' => [
          'color' => $color === '#000000' ? '' : $color,
        ],
      ];
    }
    return [
      'data' => $traces,
      'layout' => $layout + [
        'title' => [
          'text' => $options['title'] ?? new TranslatableMarkup('Violin Plot'),
        ],
        'annotations' => [
          [
            'text' => $options['description'],
            'showarrow' => FALSE,
            'xref' => 'paper',
            'yref' => 'paper',
            'y' => -0.1,
            'xanchor' => 'center',
            'yanchor' => 'top',
          ],
        ],
        'barmode' => !empty($options['stacked']) ? 'stack' : 'group',
      ],
    ];
  }

}
