<?php

namespace Drupal\plotly\Plugin\views\style;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\plotly\PlotlyChartPluginManager;
use Drupal\views\Attribute\ViewsStyle;
use Drupal\views\Plugin\views\style\StylePluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plotly.js chart style plugin.
 *
 * @ingroup views_style_plugins
 */
#[ViewsStyle(
  id: "plotly_chart",
  title: new TranslatableMarkup("Plotly chart"),
  help: new TranslatableMarkup("Render a chart of your data with Plotly."),
  theme: "views_style_plotly_chart",
  display_types: ["normal"],
)]
class Plotly extends StylePluginBase {

  /**
   * Whether this style uses a row plugin.
   *
   * @var bool
   */
  protected $usesRowPlugin = TRUE;

  /**
   * Constructs a Plugin object.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, public ?PlotlyChartPluginManager $plotlyChart = NULL) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('plugin.manager.views_plotly_chart'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions(): array {
    $options = parent::defineOptions();
    $options['type'] = ['default' => 'bar'];
    $options['webgl'] = ['default' => FALSE];
    $options['label'] = ['default' => ''];
    $options['parent'] = ['default' => ''];
    $options['direct_labels'] = ['default' => TRUE];
    $options['text'] = ['default' => ''];
    $options['group'] = ['default' => ''];
    $options['layout'] = [
      'default' => [
        'config_charts' => [
          'horizontal' => FALSE,
          'stacked' => FALSE,
          'hoverinfo' => 'value',
        ],
        'legend' => [
          'showlegend' => TRUE,
          'title' => '',
          'orientation' => 'v',
          'side' => 'top right',
        ],
        'size' => [
          'autosize' => TRUE,
          'width' => 600,
          'height' => 400,
        ],
        'format' => [
          'tickformatxaxis' => '',
          'tickprefixxaxis' => '',
          'tickformatyaxis' => '',
          'tickprefixyaxis' => '',
        ],
      ],
    ];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state): void {
    // Get all options.
    $options = [];
    $definitions = $this->plotlyChart->getDefinitions();
    foreach ($definitions as $name => $definition) {
      $cat = (string) $definition['category'];
      $options[$cat][$name] = $definition['label'];
    }

    // Field label.
    $field_labels = [];
    $fields = $this->displayHandler->getHandlers('field');
    $labels = $this->displayHandler->getFieldLabels();
    foreach ($fields as $field_name => $field) {
      if (empty($field->options["exclude"])) {
        $field_labels[$field_name] = $labels[$field_name];
      }
    }
    $example = '';
    $type = $this->options['type'];
    if (isset($definitions[$type])) {
      $example = [
        '#type' => 'link',
        '#title' => $this->t('Example: @description', ['@description' => $definitions[$type]['description']]),
        '#url' => Url::fromUri($definitions[$type]['example']),
        '#attributes' => [
          'target' => '_blank',
          'rel' => 'noopener noreferrer',
        ],
      ];
    }
    $form['type'] = [
      '#type' => 'select',
      '#options' => $options,
      '#title' => $this->t('Chart type'),
      '#description' => $example,
      '#default_value' => $type,
    ];
    $form['webgl'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use WebGL'),
      '#description' => $this->t('Use for large data.'),
      '#default_value' => $this->options['webgl'],
    ];

    $labelText = match ($type) {
      'ohlc', 'time_series' => $this->t('Date'),
      default => $this->t('Field for label')
    };
    $form['label'] = [
      '#type' => 'select',
      '#options' => $field_labels,
      '#empty_option' => $this->t('- None -'),
      '#title' => $labelText,
      '#description' => $this->t('Some type chart is x axis.'),
      '#default_value' => $this->options['label'],
    ];

    $labelText = match ($type) {
      'candlestick' => $this->t('Open'),
      'heatmap', 'contour' => $this->t('y axis'),
      'line3d', 'scatter3d', 'mesh3d', 'ribbon' => $this->t('z axis'),
      default => $this->t('Field for text or z axis in 3d mode')
    };

    $form['text'] = [
      '#type' => 'select',
      '#options' => $field_labels,
      '#empty_option' => $this->t('- None -'),
      '#title' => $labelText,
      '#description' => $this->t('Some type 3D chart require a z axis.<br/> Values field in sankey, sunburst chart'),
      '#default_value' => $this->options['text'],
    ];

    $form['group'] = [
      '#type' => 'select',
      '#options' => $field_labels,
      '#empty_option' => $this->t('- None -'),
      '#title' => $this->t('Field for group'),
      '#description' => $this->t('Group field'),
      '#default_value' => $this->options['group'],
    ];

    $sizeText = match ($type) {
      'candlestick' => $this->t('High'),
      'line3d' => $this->t('y axis'),
      default => $this->t('Size')
    };
    $form['size'] = [
      '#type' => 'select',
      '#options' => $field_labels,
      '#empty_option' => $this->t('- None -'),
      '#title' => $sizeText,
      '#description' => $this->t('Target field in sankey'),
      '#default_value' => $this->options['size'],
      '#states' => [
        'visible' => [
          [':input[name="style_options[type]"]' => ['value' => 'bubble']],
          [':input[name="style_options[type]"]' => ['value' => 'sankey']],
          [':input[name="style_options[type]"]' => ['value' => 'ohlc']],
          [':input[name="style_options[type]"]' => ['value' => 'line3d']],
          [':input[name="style_options[type]"]' => ['value' => 'candlestick']],
        ],
      ],
    ];
    $parentText = match ($type) {
      'candlestick' => $this->t('Low'),
      'line3d' => $this->t('Color'),
      default => $this->t('Field for parent')
    };
    $form['parent'] = [
      '#type' => 'select',
      '#options' => $field_labels,
      '#empty_option' => $this->t('- None -'),
      '#title' => $parentText,
      '#description' => $this->t('Use for type: treemaps, sunburst <br/> Source field in sankey'),
      '#default_value' => $this->options['parent'],
      '#states' => [
        'visible' => [
          [':input[name="style_options[type]"]' => ['value' => 'bubble']],
          [':input[name="style_options[type]"]' => ['value' => 'sankey']],
          [':input[name="style_options[type]"]' => ['value' => 'sunburst']],
          [':input[name="style_options[type]"]' => ['value' => 'treemaps']],
          [':input[name="style_options[type]"]' => ['value' => 'ohlc']],
          [':input[name="style_options[type]"]' => ['value' => 'line3d']],
          [':input[name="style_options[type]"]' => ['value' => 'candlestick']],
        ],
      ],
    ];
    $form['close'] = [
      '#type' => 'select',
      '#options' => $field_labels,
      '#empty_option' => $this->t('- None -'),
      '#title' => $this->t('Close'),
      '#description' => $this->t('Close field in candlestick chart'),
      '#default_value' => $this->options['close'],
      '#states' => [
        'visible' => [
          [':input[name="style_options[type]"]' => ['value' => 'ohlc']],
          [':input[name="style_options[type]"]' => ['value' => 'candlestick']],
        ],
      ],
    ];

    // Layout settings.
    $form['layout'] = [
      '#type' => 'details',
      '#title' => $this->t('Layout settings'),
      '#open' => FALSE,
      '#tree' => TRUE,

      'config_charts' => [
        '#type' => 'details',
        '#title' => $this->t('Config Charts'),
        '#open' => FALSE,
        '#tree' => TRUE,

        // Vị trí tiêu đề.
        'horizontal' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Horizontal charts'),
          '#description' => $this->t('Horizontal charts.'),
          '#default_value' => $this->options['layout']['config_charts']['horizontal'],
          '#states' => [
            'visible' => [
              [':input[name="style_options[type]"]' => ['value' => 'bargroup']],
              [':input[name="style_options[type]"]' => ['value' => 'linesgroup']],
              [':input[name="style_options[type]"]' => ['value' => 'dot']],
              [':input[name="style_options[type]"]' => ['value' => 'lines']],
            ],
          ],
        ],
        'stacked' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Stacked Chart'),
          '#description' => $this->t('Display stacked chart.'),
          '#default_value' => $this->options['layout']['config_charts']['stacked'],
          '#states' => [
            'visible' => [
              [':input[name="style_options[type]"]' => ['value' => 'bargroup']],
            ],
          ],
        ],
        'hoverinfo' => [
          '#type' => 'select',
          '#title' => $this->t('Pie'),
          '#default_value' => $this->options['layout']['config_charts']['hoverinfo'],
          '#options' => [
            'label' => $this->t('Label'),
            'percent' => $this->t('Percent'),
            'name' => $this->t('Name'),
            'value' => $this->t('Value'),
          ],
          '#states' => [
            'visible' => [
              [':input[name="style_options[type]"]' => ['value' => 'pie']],
              [':input[name="style_options[type]"]' => ['value' => 'bargroup']],
              [':input[name="style_options[type]"]' => ['value' => 'linesgroup']],
            ],
          ],
        ],
      ],
      // Legend.
      'legend' => [
        '#type' => 'details',
        '#title' => $this->t('Legend'),
        '#open' => FALSE,
        '#tree' => TRUE,
        // Show legend.
        'showlegend' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Show legend'),
          '#default_value' => !($this->options['layout']['legend']['showlegend'] == 0),
        ],
        'title' => [
          '#type' => 'textfield',
          '#title' => $this->t('Title'),
          '#default_value' => $this->options['layout']['legend']['title'],
        ],
        'orientation' => [
          '#type' => 'select',
          '#title' => $this->t('Orientation'),
          '#default_value' => $this->options['layout']['legend']['orientation'],
          '#options' => [
            'v' => $this->t('Vertical'),
            'h' => $this->t('Horizontal'),
          ],
        ],
        'side' => [
          '#type' => 'select',
          '#title' => $this->t('Side'),
          '#default_value' => $this->options['layout']['legend']['side'],
          '#options' => [
            'top' => $this->t('Top'),
            'left' => $this->t('Left'),
            'top left' => $this->t('Top left'),
            'top center' => $this->t('Top center'),
            'top right' => $this->t('Top right'),
          ],
        ],
      ],

      // Size.
      'size' => [
        '#type' => 'details',
        '#title' => $this->t('Size'),
        '#open' => FALSE,
        '#tree' => TRUE,
        'autosize' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Auto size'),
          '#default_value' => $this->options['layout']['size']['autosize'] == 0 ? FALSE : TRUE,
        ],
        'width' => [
          '#type' => 'number',
          '#title' => $this->t('Width'),
          '#default_value' => $this->options['layout']['size']['width'],
        ],
        'height' => [
          '#type' => 'number',
          '#title' => $this->t('Height'),
          '#default_value' => $this->options['layout']['size']['height'],
        ],
      ],
      // Format
      'format' => [
        '#type' => 'details',
        '#title' => $this->t('Format'),
        '#open' => FALSE,
        '#tree' => TRUE,
        'tickformatxaxis' => [
          '#type' => 'textfield',
          '#title' => $this->t('Tick format xaxis'),
          '#default_value' => $this->options['layout']['format']['tickformatxaxis'],
        ],
        'tickprefixxaxis' => [
          '#type' => 'textfield',
          '#title' => $this->t('Tick prefix xaxis'),
          '#default_value' => $this->options['layout']['format']['tickprefixxaxis'],
        ],
        'tickformatyaxis' => [
          '#type' => 'textfield',
          '#title' => $this->t('Tick format yaxis'),
          '#default_value' => $this->options['layout']['format']['tickformatyaxis'],
        ],
        'tickprefixyaxis' => [
          '#type' => 'textfield',
          '#title' => $this->t('Tick prefix yaxis'),
          '#default_value' => $this->options['layout']['format']['tickprefixyaxis'],
        ],
      ],
    ];

    // Per-field settings.
    $form['fields'] = [
      '#type' => 'details',
      '#title' => $this->t('Per-field settings'),
      '#open' => FALSE,
      '#tree' => TRUE,
    ];

    foreach ($field_labels as $field_name => $label) {
      $form['fields'][$field_name] = [
        '#type' => 'details',
        '#title' => $label,
        '#open' => FALSE,
        '#tree' => TRUE,
        '#states' => [
          'invisible' => [
            ':input[name="style_options[label]"]' => ['value' => $field_name],
          ],
        ],
      ];

      $form['fields'][$field_name]['color'] = [
        '#type' => 'color',
        '#title' => $this->t('Color'),
        '#default_value' => $this->options['fields'][$field_name]['color'] ?? '#000000',
      ];

      $form['fields'][$field_name]['display_type'] = [
        '#type' => 'select',
        '#options' => [
          'bar' => $this->t('Bar'),
          'scatter' => $this->t('Scatter'),
        ],
        '#empty_option' => $this->t('- None -'),
        '#title' => $this->t('Display type'),
        '#default_value' => $this->options['fields'][$field_name]['display_type'] ?? 'bar',
        '#states' => [
          'visible' => [
            [
              ':input[name="style_options[type]"]' => ['value' => 'bargroup'],
            ],
            [
              ':input[name="style_options[type]"]' => ['value' => 'linesgroup'],
            ],
            [
              ':input[name="style_options[type]"]' => ['value' => 'bar'],
            ],
            [
              ':input[name="style_options[type]"]' => ['value' => 'scatter'],
            ],
            [
              ':input[name="style_options[type]"]' => ['value' => 'candlestick'],
            ],
          ],
        ],
      ];

      $form['fields'][$field_name]['size'] = [
        '#type' => 'number',
        '#title' => $this->t('Size'),
        '#default_value' => $this->options['fields'][$field_name]['size'] ?? 10,
        '#min' => 1,
        '#max' => 100,
        '#step' => 1,
        '#states' => [
          'visible' => [
            [
              ':input[name="style_options[type]"]' => ['value' => 'dot'],
            ],
            [
              ':input[name="style_options[type]"]' => ['value' => 'scatter'],
            ],
          ],
        ],
      ];

      $form['fields'][$field_name]['size'] = [
        '#type' => 'number',
        '#title' => $this->t('Size'),
        '#default_value' => $this->options['fields'][$field_name]['size'] ?? 10,
        '#min' => 1,
        '#max' => 100,
        '#step' => 1,
        '#states' => [
          'visible' => [
            [
              ':input[name="style_options[type]"]' => ['value' => 'pie'],
            ],
            [
              ':input[name="style_options[type]"]' => ['value' => 'dot'],
            ],
            [
              ':input[name="style_options[type]"]' => ['value' => 'scatter'],
            ],
          ],
        ],
      ];
      $form['fields'][$field_name]['dash'] = [
        '#type' => 'select',
        '#title' => $this->t('Dash'),
        '#default_value' => $this->options['fields'][$field_name]['dash'] ?? 'solid',
        '#options' => [
          'solid' => $this->t('Solid'),
          'dashdot' => $this->t('Dash dot'),
          'dot' => $this->t('Dot'),
        ],
        '#empty_option' => $this->t('- None -'),
        '#states' => [
          'visible' => [
            [
              ':input[name="style_options[type]"]' => ['value' => 'lines'],
            ],
            [
              ':input[name="style_options[type]"]' => ['value' => 'scatter'],
            ],
          ],
        ],
      ];
      $form['fields'][$field_name]['shape'] = [
        '#type' => 'select',
        '#title' => $this->t('Shape'),
        '#default_value' => $this->options['fields'][$field_name]['shape'] ?? 'solid',
        '#options' => [
          'linear' => $this->t('Linear'),
          'spline' => $this->t('Spline'),
          'vh' => $this->t('Vertical horizontal'),
          'vhv' => $this->t('Vertical horizontal vertical'),
          'hv' => $this->t('Horizontal vertical'),
          'hvh' => $this->t('Horizontal vertical horizontal'),
        ],
        '#empty_option' => $this->t('- None -'),
        '#states' => [
          'visible' => [
            ':input[name="style_options[type]"]' => ['value' => 'lines'],
          ],
        ],
      ];

      $form['fields'][$field_name]['mode'] = [
        '#type' => 'select',
        '#options' => [
          'lines' => $this->t('Lines'),
          'markers' => $this->t('Markers'),
          'lines+markers' => $this->t('Lines + Markers'),
        ],
        '#empty_option' => $this->t('- None -'),
        '#title' => $this->t('Mode'),
        '#default_value' => $this->options['fields'][$field_name]['mode'] ?? 'lines',
        '#states' => [
          'visible' => [
            [
              ':input[name="style_options[type]"]' => ['value' => 'lines'],
            ],
            [
              ':input[name="style_options[type]"]' => ['value' => 'scatter'],
            ],
          ],
        ],
      ];

      $form['fields'][$field_name]['fill'] = [
        '#type' => 'select',
        '#options' => [
          'tozeroy' => $this->t('From line to 0 (along Y axis).'),
          'tozerox' => $this->t('From line to 0 (along X axis).'),
          'tonexty' => $this->t('Fill between 2 lines along the Y axis.'),
          'tonextx' => $this->t('Fill between 2 lines along the X axis.'),
          'toself' => $this->t('Connects the endpoints of the trace .'),
        ],
        '#empty_option' => $this->t('- None -'),
        '#title' => $this->t('Fill'),
        '#default_value' => $this->options['fields'][$field_name]['fill'] ?? NULL,
        '#states' => [
          'visible' => [
            [':input[name="style_options[type]"]' => ['value' => 'lines']],
            [':input[name="style_options[type]"]' => ['value' => 'filledArea']],
          ],
        ],
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $view = $this->view;
    foreach ($view->field as $field_name => $field) {
      if (empty($field->options["exclude"])) {
        $label = $field->options['label'];
        if (empty($label)) {
          $label = $field->configuration['title'] ?? '';
        }
        $this->options['fields'][$field_name]['name'] = $label;
      }
      else {
        unset($this->options['fields'][$field_name]);
      }
    }

    $this->options['title'] = $view->getTitle();
    $this->options['description'] = $this->view->storage->get('description');
    $plugin = $this->plotlyChart->createInstance($this->options['type'], []);
    $config = $plugin->data($view->result, $this->options, $view, $this->standarLayout());
    $libraries = [
      'plotly/plotly',
    ];
    if (!empty($this->options['webgl'])) {
      $libraries[] = 'plotly/virtual-webgl';
    }
    return [
      '#theme' => 'views_style_plotly_chart',
      '#view' => $view,
      '#config' => $config,
      '#attributes' => [
        'id' => 'plotly-chart-' . $view->id() . '-' . $view->current_display,
        'class' => 'plotly-chart',
      ],
      '#attached' => [
        'library' => $libraries,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function standarLayout() {
    return [
      'showlegend' => !($this->options['layout']['legend']['showlegend'] == 0),
      'legend' => [
        'title' => $this->options['layout']['legend']['title'],
        'orientation' => $this->options['layout']['legend']['orientation'],
        'side' => $this->options['layout']['legend']['side'],
      ],
      'size' => [
        'autosize' => !($this->options['layout']['size']['autosize'] == 0),
        'width' => $this->options['layout']['size']['width'],
        'height' => $this->options['layout']['size']['height'],
      ],
      'format' => [
        'tickformatxaxis' => $this->options['layout']['format']['tickformatxaxis'],
        'tickprefixxaxis' => $this->options['layout']['format']['tickprefixxaxis'],
        'tickformatyaxis' => $this->options['layout']['format']['tickformatyaxis'],
        'tickprefixyaxis' => $this->options['layout']['format']['tickprefixyaxis'],
      ],
    ];
  }

}
