<?php

declare(strict_types=1);

namespace Drupal\Tests\pluggable_entity_view_builder\Kernel;

use Drupal\entity_test\Entity\EntityTestMul;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\Tests\EntityViewTrait;
use Drupal\Tests\node\Traits\ContentTypeCreationTrait;
use Drupal\Tests\node\Traits\NodeCreationTrait;
use Drupal\user\Entity\Role;
use Drupal\user\RoleInterface;

/**
 * Tests buildEntitiesWithViewModes methods.
 *
 * @group pluggable_entity_view_builder
 */
class MultipleViewModesTest extends EntityKernelTestBase {

  use EntityViewTrait {
    buildEntityView as drupalBuildEntityView;
  }
  use ContentTypeCreationTrait;
  use NodeCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'entity_test',
    'pluggable_entity_view_builder',
    'pluggable_entity_view_builder_test',
    'system',
    'user',
  ];

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('entity_test_mul');
    $this->installEntitySchema('date_format');
    $this->installConfig('user');

    // Allow anonymous to access published content.
    $role = Role::load(RoleInterface::ANONYMOUS_ID);
    $role->grantPermission('access content');
    $role->grantPermission('view test entity');
    $role->save();

    $this->renderer = $this->container->get('renderer');

    // Create bundles for entity_test_mul.
    // We want PHPstan to work also on Drupal 10, so we ignore the deprecation
    // warning for entity_test_create_bundle().
    // @phpstan-ignore-next-line
    entity_test_create_bundle('bundle_1', 'Bundle 1', 'entity_test_mul');
    // @phpstan-ignore-next-line
    entity_test_create_bundle('bundle_2', 'Bundle 2', 'entity_test_mul');
    // @phpstan-ignore-next-line
    entity_test_create_bundle('bundle_3', 'Bundle 3', 'entity_test_mul');
  }

  /**
   * Test buildMultipleViewModes.
   */
  public function testBuildReferencedEntitiesWithViewModes(): void {
    // Create entity_test_mul entities with different bundles.
    $entity_bundle_1 = EntityTestMul::create([
      'type' => 'bundle_1',
      'name' => 'Entity Bundle 1',
    ]);
    $entity_bundle_1->save();

    $entity_bundle_2 = EntityTestMul::create([
      'type' => 'bundle_2',
      'name' => 'Entity Bundle 2',
    ]);
    $entity_bundle_2->save();

    $entity_bundle_3 = EntityTestMul::create([
      'type' => 'bundle_3',
      'name' => 'Entity Bundle 3',
    ]);
    $entity_bundle_3->save();

    // Create a main entity_test_mul that references entities with different
    // bundles. Use default bundle for main entity.
    $main_entity = EntityTestMul::create([
      'name' => 'Main Entity with Multiple View Modes',
      'field_reference' => [
        $entity_bundle_1->id(),
        $entity_bundle_2->id(),
        $entity_bundle_3->id(),
      ],
    ]);
    $main_entity->save();

    // Build entity using the new buildMultipleViewModes method.
    $build = $this->drupalBuildEntityView($main_entity, 'multiple_view_modes');
    $output = (string) $this->renderer->renderRoot($build);
    $this->setRawContent($output);

    // Assert the wrapper class is present.
    $this->assertRaw('<div class="entity-test-build-referenced-entities-with-view-modes">');

    // Assert main entity is rendered.
    $this->assertRaw('Main Entity with Multiple View Modes');

    // Assert bundle_1 entity is rendered using 'card' view mode.
    $this->assertRaw('<div class="entity-test-build-view-mode-card">');
    $this->assertRaw('Card: Entity Bundle 1');

    // Assert bundle_2 entity is rendered using 'teaser' view mode.
    $this->assertRaw('<div class="entity-test-build-view-mode-teaser">');
    $this->assertRaw('Teaser: Entity Bundle 2');

    // Assert bundle_3 entity is rendered using 'full' view mode.
    $this->assertRaw('<div class="entity-test-build-view-mode-full">');
    $this->assertRaw('Full: Entity Bundle 3');
  }

  /**
   * Test buildReferencedEntitiesWithViewModes with empty reference field.
   */
  public function testBuildReferencedEntitiesWithViewModesEmpty(): void {
    // Create main entity without references.
    $main_entity = EntityTestMul::create([
      'name' => 'Main Entity No Refs',
    ]);
    $main_entity->save();

    // Build entity using the buildMultipleViewModes method.
    $build = $this->drupalBuildEntityView($main_entity, 'multiple_view_modes');
    $output = (string) $this->renderer->renderRoot($build);
    $this->setRawContent($output);

    // Assert the wrapper class is NOT present since no references exist.
    $this->assertNoRaw('<div class="entity-test-build-referenced-entities-with-view-modes">');

    // Assert only the main entity name is rendered.
    $this->assertRaw('Main Entity No Refs');
  }

}
