<?php

namespace Drupal\plugin_configuration_field\Element;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Attribute\FormElement;
use Drupal\Core\Render\Element\FormElementBase;
use Drupal\plugin_configuration_field\Plugin\PluginEntityInterface;

/**
 * Provides a form element for configuring plugins.
 */
#[FormElement(
  id: "plugin_configuration_field",
)]
class PluginConfigurationField extends FormElementBase {

  use PluginConfigurationFieldTrait;

  /**
   * {@inheritdoc}
   */
  public function getInfo(): array {
    $class = get_class($this);
    return [
      '#input' => TRUE,
      '#tree' => TRUE,
      '#plugin_type' => NULL,
      '#plugin_id' => NULL,
      '#entity' => NULL,
      '#enforce_unique_parents' => TRUE,
      '#default_value' => [],
      '#process' => [
        [$class, 'attachElementSubmit'],
        [$class, 'processPluginConfigurationField'],
        [$class, 'processAjaxForm'],
      ],
      '#element_validate' => [
        [$class, 'validateElementSubmit'],
        [$class, 'validatePluginConfigurationField'],
      ],
      '#element_submit' => [
        [$class, 'submitPluginConfigurationField'],
      ],
      '#theme_wrappers' => ['container'],
    ];
  }

  /**
   * Processes the plugin configuration form element.
   *
   * @param array $element
   *   The form element to process.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param array $complete_form
   *   The complete form structure.
   *
   * @return array
   *   The processed element.
   *
   * @throws \InvalidArgumentException
   *   Thrown for missing #plugin_type or malformed #default_value properties.
   */
  public static function processPluginConfigurationField(array &$element, FormStateInterface $form_state, array &$complete_form): array {
    if (empty($element['#plugin_type'])) {
      throw new \InvalidArgumentException('The plugin_configuration_field #plugin_type property is required.');
    }
    if (!is_array($element['#default_value'])) {
      throw new \InvalidArgumentException('The plugin_configuration_field #default_value must be an array.');
    }
    if (empty($element['#plugin_id'])) {
      // A plugin hasn't been selected yet.
      return $element;
    }

    /** @var \Drupal\Core\Executable\ExecutableManagerInterface $plugin_manager */
    $plugin_manager = \Drupal::service('plugin.manager.' . $element['#plugin_type']);
    /** @var \Drupal\Core\Plugin\PluginFormInterface $plugin */
    $plugin = $plugin_manager->createInstance($element['#plugin_id'], $element['#default_value']);

    // If the plugin supports entities, pass in the entity.
    if ($element['#entity'] instanceof EntityInterface &&
      ($plugin instanceof PluginEntityInterface || method_exists($plugin, 'setEntity'))) {
      $plugin->setEntity($element['#entity']);
    }

    $element['form'] = [];
    if (!empty($element['#enforce_unique_parents'])) {
      // NestedArray::setValue() crashes when switching between two plugins
      // that share a configuration element of the same name, but not the
      // same type.
      // Configuration must be keyed by plugin ID in $form_state to prevent
      // that, either on this level, or in a parent form element.
      $element['form']['#parents'] = array_merge($element['#parents'], [$element['#plugin_id']]);
    }
    $element['form'] = $plugin->buildConfigurationForm($element['form'], $form_state);

    return $element;
  }

  /**
   * Validates the plugin configuration.
   *
   * @param array $element
   *   An associative array containing the properties of the element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param array $complete_form
   *   The complete form structure.
   */
  public static function validatePluginConfigurationField(array &$element, FormStateInterface $form_state, array &$complete_form): void {
    if (!empty($element['#plugin_id'])) {
      /** @var \Drupal\Component\Plugin\PluginManagerInterface $plugin_manager */
      $plugin_manager = \Drupal::service('plugin.manager.' . $element['#plugin_type']);
      $plugin = $plugin_manager->createInstance($element['#plugin_id'], $element['#default_value']);
      $plugin->validateConfigurationForm($element['form'], $form_state);
    }
  }

  /**
   * Submits the plugin configuration.
   *
   * @param array $element
   *   An associative array containing the properties of the element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public static function submitPluginConfigurationField(array &$element, FormStateInterface $form_state): void {
    if (!empty($element['#plugin_id'])) {
      /** @var \Drupal\Component\Plugin\PluginManagerInterface $plugin_manager */
      $plugin_manager = \Drupal::service('plugin.manager.' . $element['#plugin_type']);
      $plugin = $plugin_manager->createInstance($element['#plugin_id'], $element['#default_value']);
      $plugin->submitConfigurationForm($element['form'], $form_state);
      $form_state->setValueForElement($element, $plugin->getConfiguration());
    }
  }

}
