<?php

namespace Drupal\plugin_configuration_field\Plugin\Field\FieldType;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Core\Field\Attribute\FieldType;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\Core\TypedData\MapDataDefinition;
use Drupal\plugin_configuration_field\Plugin\Derivative\PluginConfigurationFieldItemDeriver;

/**
 * Plugin implementation of the "Plugin" field type.
 *
 * @property string $target_plugin_id
 *   The selected plugin ID.
 * @property array $target_plugin_configuration
 *   The selected plugin configuration.
 */
#[FieldType(
  id: "plugin_configuration_field",
  label: new TranslatableMarkup("Plugin configuration"),
  description: new TranslatableMarkup("Stores configuration for a selected plugin."),
  category: "plugin_configuration_field",
  default_widget: "plugin_configuration_field_select",
  default_formatter: "plugin_configuration_field_default",
  deriver: PluginConfigurationFieldItemDeriver::class,
)]
class PluginConfigurationFieldItem extends FieldItemBase implements PluginConfigurationFieldItemInterface {

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition): array {
    $properties['target_plugin_id'] = DataDefinition::create('string')
      ->setLabel(t('Plugin ID'));
    $properties['target_plugin_configuration'] = MapDataDefinition::create()
      ->setLabel(t('Plugin configuration'));

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function storageSettingsSummary(FieldStorageDefinitionInterface $storage_definition): array {
    $summary = parent::storageSettingsSummary($storage_definition);
    $type = $storage_definition->getType();
    [, $plugin_type] = explode(':', $type);
    // @todo Replace with plugin type label.
    $summary[] = new TranslatableMarkup('Reference type: @plugin_type', [
      '@plugin_type' => $plugin_type,
    ]);

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition): array {
    return [
      'columns' => [
        'target_plugin_id' => [
          'description' => 'The plugin ID.',
          'type' => 'varchar_ascii',
          'length' => 255,
          'not null' => TRUE,
        ],
        'target_plugin_configuration' => [
          'description' => 'The plugin configuration.',
          'type' => 'blob',
          'size' => 'big',
          'not null' => TRUE,
          'serialize' => TRUE,
        ],
      ],
      'indexes' => ['target_plugin_id' => ['target_plugin_id']],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function mainPropertyName(): ?string {
    return 'target_plugin_id';
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty(): bool {
    return empty($this->target_plugin_id) || $this->target_plugin_id == '_none';
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetDefinition(): array {
    return static::getPluginManager($this->getPluginType())->getDefinition($this->target_plugin_id);
  }

  /**
   * {@inheritdoc}
   */
  public function getTargetInstance(array $contexts = []): object {
    $plugin = static::getPluginManager($this->getPluginType())->createInstance($this->target_plugin_id, $this->target_plugin_configuration);
    // Just because the plugin is context aware, we cannot guarantee the
    // plugin manager sets them. So we apply the context mapping as well.
    if (!empty($contexts)) {
      \Drupal::service('context.handler')->applyContextMapping($plugin, $contexts);
    }

    return $plugin;
  }

  /**
   * {@inheritdoc}
   */
  public function setValue(mixed $values, $notify = TRUE): void {
    if (isset($values)) {
      $values += [
        'target_plugin_configuration' => [],
      ];
    }

    parent::setValue($values, $notify);
  }

  /**
   * Gets the plugin type.
   *
   * @return string
   *   The plugin type.
   */
  protected function getPluginType(): string {
    return $this->getPluginDefinition()['plugin_type'];
  }

  /**
   * Gets the plugin manager.
   *
   * @return \Drupal\Component\Plugin\PluginManagerInterface
   *   The plugin manager.
   */
  protected static function getPluginManager(string $plugin_type): PluginManagerInterface {
    return \Drupal::service("plugin.manager.$plugin_type");
  }

}
