<?php

namespace Drupal\plugin_configuration_field\Plugin\Field\FieldWidget;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the "Plugin select" widget.
 */
#[FieldWidget(
  id: "plugin_configuration_field_select",
  label: new TranslatableMarkup("Plugin select"),
  description: new TranslatableMarkup("A widget to select a plugin and configure its settings."),
  field_types: [
    "plugin_configuration_field",
  ],
)]
class PluginConfigurationFieldSelectWidget extends WidgetBase implements ContainerFactoryPluginInterface {
  /**
   * The plugin manager.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface
   */
  protected PluginManagerInterface $pluginManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    [, $plugin_type] = explode(':', $configuration['field_definition']->getType());
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->pluginManager = $container->get('plugin.manager.' . $plugin_type);

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    [, $plugin_type] = explode(':', $this->fieldDefinition->getType());

    $definitions = $this->pluginManager->getDefinitions();
    $plugins = array_map(static function ($definition) {
      return $definition['label'];
    }, $definitions);
    asort($plugins);
    $target_plugin_id_parents = array_merge($element['#field_parents'], [$items->getName(), $delta, 'target_plugin_id']);
    $target_plugin_id = NestedArray::getValue($form_state->getUserInput(), $target_plugin_id_parents);
    $target_plugin_configuration = [];
    // Fallback to the field value if #ajax hasn't been used yet.
    if ($target_plugin_id === NULL) {
      $target_plugin_id = $items[$delta]->target_plugin_id;
      $target_plugin_configuration = $items[$delta]->target_plugin_configuration ?: [];
    }
    elseif ($target_plugin_id === $items[$delta]->target_plugin_id) {
      $target_plugin_configuration = $items[$delta]->target_plugin_configuration ?: [];
    }
    // The element is required, default to the first plugin.
    if (!$target_plugin_id && $this->fieldDefinition->isRequired()) {
      $plugin_ids = array_keys($plugins);
      $target_plugin_id = reset($plugin_ids);
    }

    $ajax_wrapper_id = Html::getUniqueId('ajax-wrapper');
    $element = [
      '#type' => 'container',
      '#tree' => TRUE,
      '#prefix' => '<div id="' . $ajax_wrapper_id . '">',
      '#suffix' => '</div>',
    ] + $element;
    $element['target_plugin_id'] = [
      '#type' => 'select',
      '#title' => $this->fieldDefinition->getLabel(),
      '#options' => $plugins,
      '#default_value' => $target_plugin_id,
      '#required' => $this->fieldDefinition->isRequired(),
    ];
    if (!$element['target_plugin_id']['#required']) {
      $element['target_plugin_id']['#empty_value'] = '';
    }
    if (self::supportsConfiguration($definitions)) {
      $element['target_plugin_id']['#ajax'] = [
        'callback' => [get_class($this), 'ajaxRefresh'],
        'wrapper' => $ajax_wrapper_id,
      ];

      // Only build the configuration form once a plugin ID has been selected.
      if (!empty($target_plugin_id)) {
        $element['target_plugin_configuration'] = [
          '#type' => 'plugin_configuration_field',
          '#plugin_type' => $plugin_type,
          '#plugin_id' => $target_plugin_id,
          '#entity' => $items->getEntity(),
          '#default_value' => $target_plugin_configuration,
        ];
      }
    }

    return $element;
  }

  /**
   * Determines whether plugin configuration is supported.
   *
   * Supported if the plugins implement PluginFormInterface.
   *
   * @param array $definitions
   *   The available plugin definitions.
   *
   * @return bool
   *   TRUE if plugin configuration is supported, FALSE otherwise.
   */
  protected static function supportsConfiguration(array $definitions): bool {
    // The plugin manager has $this->pluginInterface, but there's no getter
    // for it, so it can't be used to check for PluginFormInterface.
    // Instead, we assume that all plugins implement the same interfaces,
    // and perform the check against the first plugin.
    $definition = reset($definitions);

    return is_subclass_of($definition['class'], PluginFormInterface::class);
  }

  /**
   * Ajax callback.
   *
   * @param array &$form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The refreshed form element.
   */
  public static function ajaxRefresh(array &$form, FormStateInterface $form_state): array {
    $element_parents = array_slice($form_state->getTriggeringElement()['#array_parents'], 0, -1);

    return NestedArray::getValue($form, $element_parents);
  }

}
