<?php

declare(strict_types=1);

namespace Drupal\Tests\plural_serialization\Kernel;

use Drupal\Component\Gettext\PoItem;
use Drupal\Core\Config\ConfigImporter;
use Drupal\Core\Config\StorageComparer;
use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\config_filter\Kernel\ConfigStorageTestTrait;

/**
 * @coversDefaultClass \Drupal\plural_serialization\EventSubscriber\PluralSerializationConfigSubscriber
 * @group plural_serialization
 */
class PluralSerializationTest extends KernelTestBase {

  use ConfigStorageTestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'plural_serialization',
    'plural_serialization_test',
  ];

  /**
   * @covers ::onExport
   * @covers ::onImport
   * @dataProvider exportAndImportProvider
   *
   * @param string $binaryEncodedPlural
   *   The binary string. Acts also as expected strung on import.
   * @param array $expected
   *   The array expected on export.
   */
  public function testExportAndImport(string $binaryEncodedPlural, array $expected): void {
    $storage = $this->container->get('config.storage');
    $export = $this->container->get('config.storage.export');
    $sync = $this->container->get('config.storage.sync');
    $configName = 'plural_serialization_test.settings';

    // Create a config with plural variants.
    $this->config($configName)->set('label', $binaryEncodedPlural)->save();

    // Export to config sync.
    foreach ($export->listAll() as $name) {
      $sync->write($name, $export->read($name));
    }

    // Check that exported plural variants are array instead of binary string.
    $this->assertSame($expected, $sync->read($configName)['label']);

    // Set up the config importer.
    $comparer = new StorageComparer(
      $this->container->get('config.import_transformer')->transform($this->container->get('config.storage.sync')),
      $this->container->get('config.storage')
    );
    $importer = new ConfigImporter(
      $comparer->createChangelist(),
      $this->container->get('event_dispatcher'),
      $this->container->get('config.manager'),
      $this->container->get('lock'),
      $this->container->get('config.typed'),
      $this->container->get('module_handler'),
      $this->container->get('module_installer'),
      $this->container->get('theme_handler'),
      $this->container->get('string_translation'),
      $this->container->get('extension.list.module'),
      $this->container->get('extension.list.theme')
    );

    // Import from config sync.
    $importer->reset()->import();

    // Check that plural variants were imported as a binary string.
    $this->assertSame($binaryEncodedPlural, $storage->read($configName)['label']);
  }

  /**
   * Provides data for ::testExportAndImport() test.
   *
   * @return iterable<array-key, array{string, list<string>}>
   *   Test cases.
   */
  public static function exportAndImportProvider(): iterable {
    yield 'full' => [
      'singular' . PoItem::DELIMITER . '1st plural' . PoItem::DELIMITER . '2nd plural',
      ['singular', '1st plural', '2nd plural'],
    ];

    yield 'missing singular' => [
      PoItem::DELIMITER . '1st plural' . PoItem::DELIMITER . '2nd plural',
      ['', '1st plural', '2nd plural'],
    ];

    yield 'only singular' => [
      'singular',
      ['singular'],
    ];

    yield 'missing 1st plural' => [
      'singular' . PoItem::DELIMITER . PoItem::DELIMITER . '2nd plural',
      ['singular', '', '2nd plural'],
    ];

    yield 'missing last plural' => [
      'singular' . PoItem::DELIMITER . '1st plural',
      ['singular', '1st plural'],
    ];

    yield 'last plural empty' => [
      'singular' . PoItem::DELIMITER . '1st plural ' . PoItem::DELIMITER,
      ['singular', '1st plural ', ''],
    ];

    yield 'empty' => [
      '',
      [],
    ];

    yield 'spaces' => [
      '  ',
      ['  '],
    ];

    yield 'leading and trailing spaces' => [
      ' singular ' . PoItem::DELIMITER . ' 1st plural ' . PoItem::DELIMITER . ' 2nd plural ',
      [' singular ', ' 1st plural ', ' 2nd plural '],
    ];

  }

}
