<?php

declare(strict_types=1);

namespace Drupal\plural_serialization\EventSubscriber;

use Drupal\Component\Gettext\PoItem;
use Drupal\Core\Config\ConfigEvents;
use Drupal\Core\Config\StorageTransformEvent;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\TypedData\TraversableTypedDataInterface;
use Drupal\Core\TypedData\TypedDataInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Listens to config import and export.
 */
class PluralSerializationConfigSubscriber implements EventSubscriberInterface {

  public function __construct(
    protected readonly TypedConfigManagerInterface $typedConfig,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      ConfigEvents::STORAGE_TRANSFORM_EXPORT => 'onExport',
      ConfigEvents::STORAGE_TRANSFORM_IMPORT => 'onImport',
    ];
  }

  /**
   * Applies transformations on export.
   *
   * @param \Drupal\Core\Config\StorageTransformEvent $event
   *   The event.
   */
  public function onExport(StorageTransformEvent $event): void {
    $this->transform('export', $event);
  }

  /**
   * Applies transformations on import.
   *
   * @param \Drupal\Core\Config\StorageTransformEvent $event
   *   The event.
   */
  public function onImport(StorageTransformEvent $event): void {
    $this->transform('import', $event);
  }

  /**
   * Applies transformations on export and import.
   *
   * @param string $direction
   *   Options: 'export', 'import'.
   * @param \Drupal\Core\Config\StorageTransformEvent $event
   *   The transformation event.
   */
  protected function transform(string $direction, StorageTransformEvent $event): void {
    $storage = $event->getStorage();

    foreach ($storage->listAll() as $name) {
      $data = $original = $storage->read($name);
      $element = $this->typedConfig->get($name);

      $this->doTransform($name, $data, $element, $direction);

      if ($data !== $original) {
        $storage->write($name, $data);
      }
    }
  }

  /**
   * Applies transformation recursively.
   *
   * @param string $name
   *   The config name.
   * @param mixed $data
   *   Config data passed by reference.
   * @param \Drupal\Core\TypedData\TypedDataInterface $element
   *   The config typed element.
   * @param string $direction
   *   Options: 'export', 'import'.
   */
  protected function doTransform(string $name, mixed &$data, TypedDataInterface $element, string $direction): void {
    if ($element instanceof TraversableTypedDataInterface) {
      foreach ($element as $key => $childElement) {
        // This is a mapping/sequence, descent to a deeper level.
        $this->doTransform($name, $data[$key], $childElement, $direction);
      }
    }
    else {
      $definition = $element->getDataDefinition();
      if ($definition->getDataType() === 'plural_label') {
        if ($direction === 'export' && is_string($data)) {
          $data = $data === '' ? [] : explode(PoItem::DELIMITER, $data);
        }
        elseif ($direction === 'import' && is_array($data)) {
          $data = implode(PoItem::DELIMITER, $data);
        }
        else {
          throw new \InvalidArgumentException();
        }
      }
    }
  }

}
