<?php

declare(strict_types=1);

namespace Drupal\Tests\plural_serialization\Kernel;

use Drupal\Component\Gettext\PoItem;
use Drupal\Core\Config\ConfigImporter;
use Drupal\Core\Config\StorageComparer;
use Drupal\Core\Config\StorageInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\language\Entity\ConfigurableLanguage;

/**
 * @coversDefaultClass \Drupal\plural_serialization\EventSubscriber\PluralSerializationConfigSubscriber
 * @group plural_serialization
 */
class PluralSerializationTest extends KernelTestBase {

  /**
   * Test config name.
   */
  private const CONFIG = 'plural_serialization_test.settings';

  /**
   * Test config property.
   */
  private const PROPERTY = 'label';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'language',
    'plural_serialization',
    'plural_serialization_test',
  ];

  /**
   * @covers ::onExport
   * @covers ::onImport
   * @dataProvider exportAndImportProvider
   *
   * @param string $binaryEn
   *   The binary string in English. Acts also as expected string on import.
   * @param array $arrayEn
   *   The expected array on export in English.
   * @param string $binaryRo
   *   The binary string in Romanian. Acts also as expected string on import.
   * @param array $arrayRo
   *   The expected array on export in Romanian.
   */
  public function testExportAndImport(string $binaryEn, array $arrayEn, string $binaryRo, array $arrayRo): void {
    ConfigurableLanguage::createFromLangcode('ro')->save();

    $storage = $this->container->get('config.storage');
    $export = $this->container->get('config.storage.export');
    $sync = $this->container->get('config.storage.sync');

    // Create a config with plural variants and its Romanian translation.
    $this->config(self::CONFIG)->set(self::PROPERTY, $binaryEn)->save();
    $override = $this->container->get('language_manager')->getLanguageConfigOverride('ro', self::CONFIG);
    $override->set(self::PROPERTY, $binaryRo)->save();

    $collections = [StorageInterface::DEFAULT_COLLECTION, ...$storage->getAllCollectionNames()];
    foreach ($collections as $collection) {
      $source = $collection === StorageInterface::DEFAULT_COLLECTION ? $export : $export->createCollection($collection);
      $destination = $collection === StorageInterface::DEFAULT_COLLECTION ? $sync : $sync->createCollection($collection);

      // Export to config sync.
      foreach ($source->listAll() as $name) {
        $destination->write($name, $source->read($name));
      }
    }

    // Check that exported plural variants are array instead of binary string.
    $this->assertExport($arrayEn, $arrayRo, $sync);

    // Set up the config importer.
    $comparer = new StorageComparer(
      $this->container->get('config.import_transformer')->transform($this->container->get('config.storage.sync')),
      $this->container->get('config.storage')
    );
    $importer = new ConfigImporter(
      $comparer->createChangelist(),
      $this->container->get('event_dispatcher'),
      $this->container->get('config.manager'),
      $this->container->get('lock'),
      $this->container->get('config.typed'),
      $this->container->get('module_handler'),
      $this->container->get('module_installer'),
      $this->container->get('theme_handler'),
      $this->container->get('string_translation'),
      $this->container->get('extension.list.module'),
      $this->container->get('extension.list.theme')
    );

    // Import from config sync.
    $importer->reset()->import();

    // Check that plural variants were imported as a binary string.
    $this->assertImport($binaryEn, $binaryRo, $storage);
  }

  /**
   * Provides data for ::testExportAndImport() test.
   *
   * @return iterable<array-key, array{string, list<string>}>
   *   Test cases.
   */
  public static function exportAndImportProvider(): iterable {
    yield 'full' => [
      'one apple' . PoItem::DELIMITER . '@count apples',
      ['one apple', '@count apples'],
      'un măr' . PoItem::DELIMITER . '@count mere' . PoItem::DELIMITER . '@count de mere',
      ['un măr', '@count mere', '@count de mere'],
    ];

    yield 'missing singular' => [
      PoItem::DELIMITER . '@count apples',
      ['', '@count apples'],
      PoItem::DELIMITER . '@count mere' . PoItem::DELIMITER . '@count de mere',
      ['', '@count mere', '@count de mere'],
    ];

    yield 'only singular' => [
      'one apple',
      ['one apple'],
      'un măr',
      ['un măr'],
    ];

    yield 'first plural empty' => [
      'one apple' . PoItem::DELIMITER,
      ['one apple', ''],
      'un măr' . PoItem::DELIMITER . PoItem::DELIMITER . '@count de mere',
      ['un măr', '', '@count de mere'],
    ];

    yield 'missing last plural' => [
      'one apple' . PoItem::DELIMITER . '@count apples',
      ['one apple', '@count apples'],
      'un măr' . PoItem::DELIMITER . '@count mere',
      ['un măr', '@count mere'],
    ];

    yield 'last plural empty' => [
      'one apple' . PoItem::DELIMITER . '@count apples ' . PoItem::DELIMITER,
      ['one apple', '@count apples ', ''],
      'un măr' . PoItem::DELIMITER . '@count mere' . PoItem::DELIMITER,
      ['un măr', '@count mere', ''],
    ];

    yield 'empty' => [
      '',
      [],
      '',
      [],
    ];

    yield 'spaces' => [
      '  ',
      ['  '],
      '  ',
      ['  '],
    ];

    yield 'leading and trailing spaces' => [
      ' one apple ' . PoItem::DELIMITER . ' @count apples ',
      [' one apple ', ' @count apples '],
      ' un măr ' . PoItem::DELIMITER . ' @count mere ' . PoItem::DELIMITER . ' @count de mere ',
      [' un măr ', ' @count mere ', ' @count de mere '],
    ];
  }

  /**
   * Asserts that the export works.
   *
   * @param array $arrayEn
   *   The expected array on export in English.
   * @param array $arrayRo
   *   The expected array on export in Romanian.
   * @param \Drupal\Core\Config\StorageInterface $storage
   *   The config storage.
   */
  protected function assertExport(array $arrayEn, array $arrayRo, StorageInterface $storage): void {
    $this->assertSame($arrayEn, $storage->read(self::CONFIG)[self::PROPERTY]);
    $this->assertSame(
      $arrayRo,
      $storage->createCollection('language.ro')->read(self::CONFIG)[self::PROPERTY]
    );
  }

  /**
   * Asserts that the import works.
   *
   * @param string $binaryEn
   *   The binary string in English. Acts also as expected string on import.
   * @param string $binaryRo
   *   The binary string in Romanian. Acts also as expected string on import.
   * @param \Drupal\Core\Config\StorageInterface $storage
   *   The config storage.
   */
  protected function assertImport(string $binaryEn, string $binaryRo, StorageInterface $storage): void {
    $this->assertSame($binaryEn, $storage->read(self::CONFIG)[self::PROPERTY]);
    $this->assertSame(
      $binaryRo,
      $storage->createCollection('language.ro')->read(self::CONFIG)[self::PROPERTY]
    );
  }

}
