<?php

declare(strict_types=1);

namespace Drupal\plural_serialization\EventSubscriber;

use Drupal\Component\Gettext\PoItem;
use Drupal\Core\Config\ConfigEvents;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Config\StorageTransformEvent;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\TypedData\TraversableTypedDataInterface;
use Drupal\Core\TypedData\TypedDataInterface;
use Drupal\plural_serialization\Direction;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Listens to config import and export.
 */
class PluralSerializationConfigSubscriber implements EventSubscriberInterface {

  public function __construct(
    protected readonly TypedConfigManagerInterface $typedConfig,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      ConfigEvents::STORAGE_TRANSFORM_EXPORT => 'onExport',
      ConfigEvents::STORAGE_TRANSFORM_IMPORT => 'onImport',
    ];
  }

  /**
   * Applies transformations on export.
   *
   * @param \Drupal\Core\Config\StorageTransformEvent $event
   *   The event.
   */
  public function onExport(StorageTransformEvent $event): void {
    $this->transform(Direction::Export, $event);
  }

  /**
   * Applies transformations on import.
   *
   * @param \Drupal\Core\Config\StorageTransformEvent $event
   *   The event.
   */
  public function onImport(StorageTransformEvent $event): void {
    $this->transform(Direction::Import, $event);
  }

  /**
   * Applies transformations on export and import.
   *
   * @param \Drupal\plural_serialization\Direction $direction
   *   The config sync operation direction.
   * @param \Drupal\Core\Config\StorageTransformEvent $event
   *   The transformation event.
   */
  protected function transform(Direction $direction, StorageTransformEvent $event): void {
    $storage = $event->getStorage();

    $collections = [
      StorageInterface::DEFAULT_COLLECTION,
      ...$storage->getAllCollectionNames(),
    ];

    foreach ($collections as $collection) {
      $transform = $storage->createCollection($collection);

      foreach ($transform->listAll() as $name) {
        $data = $original = $transform->read($name);
        $element = $this->typedConfig->get($name);

        $this->doTransform($direction, $data, $element);

        if ($data !== $original) {
          $transform->write($name, $data);
        }
      }
    }
  }

  /**
   * Applies transformation recursively.
   *
   * @param \Drupal\plural_serialization\Direction $direction
   *   The config sync operation direction.
   * @param mixed $data
   *   Config data passed by reference.
   * @param \Drupal\Core\TypedData\TypedDataInterface $element
   *   The config typed element.
   */
  protected function doTransform(Direction $direction, mixed &$data, TypedDataInterface $element): void {
    if ($element instanceof TraversableTypedDataInterface) {
      foreach ($element as $key => $childElement) {
        // This is a mapping/sequence, try to descend to a deeper level.
        if (array_key_exists($key, $data)) {
          $this->doTransform($direction, $data[$key], $childElement);
        }
      }
    }
    else {
      $definition = $element->getDataDefinition();
      if ($definition->getDataType() === 'plural_label') {
        $data = match(TRUE) {
          $direction === Direction::Export && is_string($data) => ($data === '') ? [] : explode(PoItem::DELIMITER, $data),
          $direction === Direction::Import && is_array($data) => implode(PoItem::DELIMITER, $data),
        };
      }
    }
  }

}
