#!/bin/bash
# Check for Docker
if ! command -v docker &> /dev/null; then
    echo "Docker is required. Please install it first: https://docs.docker.com/get-docker/"
    exit 1
fi

if ! docker info &> /dev/null; then
    echo "Docker provider (e.g., Docker Desktop, Lima, Colima, Orbstack, etc) is not running."
    echo "Please start your Docker provider and try again."
    exit 1
fi

# Check for DDEV
if ! command -v ddev &> /dev/null; then
    echo "DDEV is required. Install it with: brew install ddev/ddev/ddev (macOS/Linux) or see https://ddev.readthedocs.io/en/stable/"
    exit 1
fi

# Prompt user for project directory name
read -p "Enter the project directory name (default: plus-suite): " project_name
project_name="${project_name:-plus-suite}"

# Create directory, configure, and install Drupal
mkdir "$project_name" && cd "$project_name" || exit
echo 'Configuring ddev...'
ddev config --project-type=drupal --docroot=web --create-docroot --php-version='8.3'
echo 'Starting ddev...'
ddev start

echo 'Creating drupal/recommended-project...'
ddev composer create-project drupal/recommended-project:"~11.3.0" -y
# This is an easy way to get set up for contributing
#ddev composer create drupal/recommended-project --prefer-source -y

echo 'Fetching Drush...'
ddev composer require drush/drush
echo 'Setting minimum stability...'
composer config minimum-stability dev
echo 'Setting prefer stable...'
composer config prefer-stable false
echo 'Installing Drupal...'
ddev drush site:install --account-name=admin --account-pass=admin -y
echo 'Adding the DropzoneJS repository...'
composer config repositories.dropzone '{"type": "package", "package": {"name": "enyo/dropzone", "version": "6.0.0-beta.2", "dist": {"type": "zip", "url": "https://github.com/dropzone/dropzone/releases/download/v6.0.0-beta.2/dist.zip"}, "type": "drupal-library"}}'
echo 'Adding unpack composer plugin...'
composer require drupal/core-recipe-unpack
echo 'Allow unpacking...'
composer config allow-plugins.drupal/core-recipe-unpack true
echo 'Fetching + Suite...'
ddev composer require drupal/plus_suite '1.1.x-dev' --prefer-source
echo 'Installing + Suite...'
ddev drush recipe ../recipes/plus_suite/
echo 'Unpacking plus_suite to the sites composer.json...'
composer drupal:recipe-unpack drupal/plus_suite -y
echo 'Clearing caches...'
ddev drush cr
echo 'Launching site...'
if command -v open &> /dev/null; then
    ddev drush uli | xargs open
else
    ddev launch
fi
echo 'Clearing caches...'
ddev drush cr
SHOW_ART=true
while [[ $# -gt 0 ]]; do
    case $1 in
        --no-art)
            SHOW_ART=false
            shift
            ;;
        *)
            shift
            ;;
    esac
done

if [ "$SHOW_ART" = true ] && [ $? -eq 0 ]; then
  echo "
[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m    [37m [0m[37m [0m[37m [0m[37m [0m      [37m [0m[37m [0m[37m [0m[37m.[0m[34m,[0m[34m.[0m[34m [0m[37m [0m[37m [0m[37m [0m [37m [0m[37m [0m[37m [0m     [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m   [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m  [37m [0m[37m [0m[37m [0m[37m [0m[37m,[0m[34md[0m[34m:[0m[34m.[0m[34m.[0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m  [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[36mc[0m[36mX[0m[34md[0m[34m,[0m[34m,[0m[34m.[0m[34m.[0m[37m [0m[37m [0m [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m  [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m.[0m[37m'[0m[36mO[0m[37mN[0m[34mx[0m[34m;[0m[34m;[0m[34m,[0m[34m,[0m[34m'[0m[34m.[0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m    [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m.[0m[37m,[0m[36mo[0m[36mK[0m[37mW[0m[36m0[0m[34ml[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m.[0m[34m.[0m[34m.[0m[37m [0m[37m [0m[37m [0m[37m [0m       [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m  [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m.[0m[37m'[0m[36mc[0m[36mx[0m[37mK[0m[37mN[0m[37mN[0m[34mO[0m[34ml[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m,[0m[34m'[0m[34m.[0m[34m.[0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
    [37m [0m[37m [0m[37m [0m[37m [0m  [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m  [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m.[0m[37m;[0m[36mx[0m[36mK[0m[37mW[0m[37mW[0m[37mN[0m[36mK[0m[36mO[0m[34mo[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m,[0m[34m,[0m[34m.[0m[34m.[0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
[37m [0m[37m [0m  [37m [0m[37m [0m[37m [0m  [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m.[0m[37m.[0m[37m;[0m[37ml[0m[37mx[0m[37m0[0m[37mN[0mM[37mM[0m[37mM[0m[37mM[0m[37mN[0m[36mK[0m[34mO[0m[34mo[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m,[0m[34m'[0m[34m.[0m[34m.[0m[34m.[0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
[37m [0m[37m [0m   [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m [37m [0m[37m [0m[37m [0m[37m [0m[37m.[0m[37m'[0m[36mo[0m[36m0[0m[37mX[0m[37mW[0mMMMMMM[37mW[0m[36mX[0m[36m0[0m[34mk[0m[34md[0m[34m:[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m,[0m[34m.[0m[34m.[0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m    [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
 [37m [0m[37m [0m[37m [0m  [37m [0m[37m [0m[37m [0m  [37m [0m[37m [0m[37m [0m[37m.[0m[37m'[0m[36mc[0m[36mx[0m[37m0[0m[37mW[0m[37mM[0m[37mM[0m[37mM[0m[37mM[0mMMM[37mW[0m[37mX[0m[36m0[0m[34mO[0m[34mx[0m[34ml[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m,[0m[34m'[0m[34m.[0m[34m.[0m[37m [0m[37m [0m[37m [0m  [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
  [37m [0m[37m [0m  [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m'[0m[36ml[0m[36mk[0m[37mK[0m[37mW[0mM[37mM[0m[37mM[0m[37mM[0m[37mM[0mMM[37mM[0m[37mW[0m[36mX[0m[34mO[0m[34mx[0m[34mo[0m[34mc[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m,[0m[34m'[0m[34m.[0m[34m.[0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
[37m [0m[37m [0m[37m [0m    [37m [0m[37m [0m[37m [0m[37m.[0m[36mo[0m[37mX[0mMMMMMM[37mM[0mM[37mW[0m[37mW[0m[37mX[0m[36mK[0m[34m0[0m[34mx[0m[34mc[0m[34m:[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m.[0m[37m [0m[37m [0m     [37m [0m[37m [0m[37m [0m
[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m.[0m[36m:[0m[36mO[0m[37mW[0mMMMMMMM[37mW[0m[37mX[0m[36mK[0m[34m0[0m[34mO[0m[34mx[0m[34mo[0m[34m:[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m.[0m[34m.[0m[37m [0m[37m [0m[37m [0m    [37m [0m
[37m [0m[37m [0m  [37m [0m[37m [0m[37m.[0m[36mc[0m[36mO[0m[37mN[0mMMMM[37mM[0m[37mW[0m[37mN[0m[37mX[0m[36mX[0m[34mO[0m[34mx[0m[34mo[0m[34ml[0m[34mc[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m'[0m[34m.[0m[34m.[0m[37m [0m[37m [0m[37m [0m [37m [0m[37m [0m
[37m [0m  [37m [0m[37m [0m[34m,[0m[34mo[0m[36mK[0m[37mW[0mW[37mW[0m[37mN[0m[36mX[0m[36mK[0m[36mK[0m[34m0[0m[34mk[0m[34mx[0m[34mo[0m[34mc[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m'[0m[34m.[0m[34m.[0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
  [37m [0m[37m [0m[34m'[0m[34ml[0m[34mO[0m[34mO[0m[34m0[0m[36m0[0m[36m0[0m[34m0[0m[34mO[0m[34mx[0m[34md[0m[34ml[0m[34mc[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m.[0m[37m [0m[37m [0m[37m [0m[37m [0m
 [37m [0m[37m [0m[34m.[0m[34m,[0m[34ml[0m[34mx[0m[34mk[0m[34mx[0m[34mx[0m[34md[0m[34mo[0m[34mc[0m[34m:[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m.[0m[37m [0m[37m [0m
[37m [0m[37m [0m[34m.[0m[34m'[0m[34m;[0m[34m;[0m[34m;[0m[34m:[0m[34m:[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m.[0m[34m.[0m[37m [0m[37m [0m
[37m [0m[34m.[0m[34m'[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m.[0m[37m.[0m[37m [0m
[34m.[0m[34m'[0m[34m,[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m.[0m[37m [0m
[34m.[0m[34m'[0m[34m,[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m.[0m[34m.[0m
[34m'[0m[34m,[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m.[0m
[34m,[0m[34m,[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m.[0m
[34m,[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m.[0m
[34m,[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34mc[0m[34mc[0m[34ml[0m[34ml[0m[34ml[0m[34mc[0m[34mc[0m[34m:[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m
[34m,[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34mc[0m[34mo[0m[34mx[0m[36m0[0m[36mX[0m[36mX[0m[36mN[0m[36mN[0m[36mN[0m[36mN[0m[36mX[0m[36mK[0m[34mO[0m[34mx[0m[34mo[0m[34mc[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m;[0m[34mc[0m[34mo[0m[34mo[0m[34ml[0m[34mc[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m
[34m,[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m:[0m[34mo[0m[34mk[0m[36mK[0m[37mN[0mWMMMMMMMMMMMM[37mW[0m[37mN[0m[36mK[0m[36mO[0m[34mx[0m[34ml[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m,[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m;[0m[34mc[0m[34mo[0m[36mO[0m[37mX[0m[37mW[0mMMM[37mN[0m[36mK[0m[34ml[0m[34m'[0m[34m'[0m[34m'[0m
[34m'[0m[34m,[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m:[0m[34mk[0m[37mW[0mMMMMMMMMMMMM[37mM[0m[37mM[0m[37mM[0m[37mM[0m[37mM[0mM[37mM[0mMM[37mW[0m[36m0[0m[34mx[0m[34ml[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m;[0m[34mo[0m[36mk[0m[37mX[0m[37mW[0mMMMM[37mM[0m[37mM[0mM[37mM[0mM[36mK[0m[34m:[0m[34m'[0m[34m.[0m
[34m.[0m[34m,[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m:[0m[34mk[0m[37mW[0mMMMMMMMMMMMMMMMMMMMMMMM[37mM[0m[37mM[0m[37mN[0m[36m0[0m[34mx[0m[34ml[0m[34m;[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m,[0m[34ml[0m[36mO[0m[36mX[0m[37mW[0mM[37mM[0m[37mM[0m[37mM[0m[37mM[0mMMMMM[37mM[0m[37mM[0m[37mW[0m[36md[0m[34m'[0m[34m.[0m
[34m.[0m[34m'[0m[34m,[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34mx[0m[37mW[0mM[37mM[0m[37mM[0mMMMMM[37mM[0m[37mM[0m[37mM[0m[37mM[0m[37mM[0m[37mM[0m[37mM[0mMMM[37mM[0m[37mM[0m[37mM[0mMMMMMMMM[37mN[0m[36m0[0m[36md[0m[34ml[0m[34mc[0m[34mc[0m[34mo[0m[36mx[0m[36mO[0mNM[37mM[0m[37mM[0m[37mM[0mMMM[37mM[0m[37mM[0m[37mM[0mMMMMM[37mW[0m[36md[0m[34m'[0m[34m.[0m
[37m [0m[34m.[0m[34m,[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34mo[0m[36mX[0mM[37mM[0mMMMMMMMMMMMMMMMMM[37mM[0m[37mM[0m[37mM[0m[37mM[0m[37mM[0mMMMMM[37mM[0m[37mM[0m[37mM[0mM[37mW[0m[37mN[0m[37mW[0mMMMMM[37mM[0mMMMMMMMMMMMMM[37mN[0m[34mc[0m[37m.[0m[37m [0m
[37m [0m[37m [0m[34m.[0m[34m,[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34mx[0m[36mW[0mMMMMMMMMMMMMMMMMMMMMMMMM[37mM[0m[37mM[0m[37mM[0m[37mM[0mM[37mW[0m[37mX[0m[36m0[0m[36md[0m[34ml[0m[34mc[0m[34ml[0m[36mo[0m[36mk[0m[37mX[0m[37mW[0mMMMMMMMMMMMMMMM[37m0[0m[34m,[0m[37m.[0m[37m [0m
[37m [0m[37m [0m[37m [0m[34m'[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34mo[0m[36mX[0mM[37mM[0mMMMMMMMMM[37mM[0m[37mM[0m[37mM[0mMMMMMMMMMM[37mM[0m[37mM[0m[37mN[0m[37m0[0m[36mx[0m[34ml[0m[34m:[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m:[0m[36mO[0m[37mW[0mMMMMMMM[37mM[0m[37mM[0m[37mM[0mMMMM[36mx[0m[37m.[0m[37m [0m[37m [0m
 [37m [0m[37m [0m[34m.[0m[34m,[0m[34m,[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m:[0m[34mx[0m[37mN[0mMMMMMMMMM[37mM[0m[37mM[0mMMMMMMMMM[37mW[0m[37mN[0m[37mK[0m[36mk[0m[34ml[0m[34m:[0m[34m,[0m[34m'[0m[34m'[0m[34m,[0m[34m;[0m[34m;[0m[34m;[0m[34m,[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m,[0m[34ml[0m[36mk[0m[37mX[0mWMMMM[37mM[0m[37mM[0m[37mM[0mMM[37mW[0m[37m0[0m[37m,[0m[37m [0m[37m [0m[37m [0m
  [37m [0m[37m [0m[34m.[0m[34m.[0m[34m'[0m[34m,[0m[34m,[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34m;[0m[34mo[0m[36mX[0mMMMMMMMMMMMMMMMM[37mN[0m[36m0[0m[36mx[0m[34mo[0m[34m:[0m[34m,[0m[34m'[0m[34m'[0m[34m,[0m[34mc[0m[36md[0m[36mO[0m[36m0[0m[36mX[0m[36mX[0m[37mX[0m[37mK[0m[36m0[0m[36mk[0m[34ml[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m;[0m[34mo[0m[36m0[0m[37mW[0mMMMMMM[37mN[0m[36md[0m[37m,[0m[37m [0m[37m [0m [37m [0m
   [37m [0m[37m [0m[34m [0m[34m.[0m[34m'[0m[34m'[0m[34m'[0m[34m,[0m[34m,[0m[34m,[0m[34m,[0m[34m,[0m[34m,[0m[34m,[0m[34m,[0m[34m,[0m[34mc[0m[36mk[0m[36m0[0m[37mX[0m[37mW[0mMMMMMMM[37mW[0m[37mN[0m[37mX[0m[36mO[0m[34mo[0m[34m:[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m,[0m[34mo[0m[36mO[0m[36mN[0m[36mN[0m[36m0[0m[36mx[0m[36md[0m[34mo[0m[34md[0m[36mx[0m[36m0[0m[37mN[0m[37mN[0m[36mx[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m;[0m[34ml[0m[36mO[0m[37mN[0mW[37mW[0m[36mX[0m[36mk[0m[36mc[0m[37m.[0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
 [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m.[0m[34m.[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m,[0m[34m;[0m[34m:[0m[34mc[0m[34mc[0m[34ml[0m[34ml[0m[34ml[0m[34ml[0m[34mc[0m[34mc[0m[34m:[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34mc[0m[36mO[0m[34mx[0m[34mo[0m[34m:[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m,[0m[34ml[0m[36mk[0m[34mx[0m[34m;[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m:[0m[34mc[0m[34m:[0m[34m,[0m[34m.[0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
 [37m [0m[37m [0m[37m [0m[37m [0m [37m [0m[37m [0m[37m [0m[37m.[0m[34m.[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m.[0m[37m.[0m[37m [0m[37m [0m[37m [0m[37m [0m
[37m [0m  [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[34m.[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m,[0m[34m:[0m[34m;[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m,[0m[34mc[0m[34m;[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m.[0m[34m.[0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
[37m [0m[37m [0m [37m [0m[37m [0m[37m [0m[37m [0m   [37m [0m[37m [0m[37m [0m[37m.[0m[34m.[0m[34m.[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34mo[0m[36mX[0m[36mX[0m[36mk[0m[36md[0m[34mc[0m[34m:[0m[34m;[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m,[0m[34m;[0m[34m:[0m[34mc[0m[36mo[0m[36mx[0m[36m0[0m[36mK[0m[34mc[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m.[0m[34m.[0m[37m [0m[37m [0m[37m [0m[37m [0m [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
[37m [0m[37m [0m[37m [0m   [37m [0m[37m [0m [37m [0m   [37m [0m[37m [0m[37m [0m[37m.[0m[34m.[0m[34m.[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m;[0m[36md[0m[36mO[0m[36m0[0m[36mX[0m[36mX[0m[36mX[0m[36mX[0m[36mK[0m[36m0[0m[36m0[0m[36m0[0m[36m0[0m[36m0[0m[36m0[0m[36mK[0m[36m0[0m[36m0[0m[36m0[0m[36m0[0m[36mO[0m[36mx[0m[34mo[0m[34m:[0m[34m'[0m[34m.[0m[34m.[0m[34m.[0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
[37m [0m   [37m [0m[37m [0m  [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m.[0m[34m.[0m[34m.[0m[34m'[0m[34m.[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m,[0m[34m:[0m[34mc[0m[34ml[0m[34mo[0m[36mo[0m[36md[0m[36md[0m[34md[0m[34mo[0m[34mo[0m[34ml[0m[34mc[0m[34m:[0m[34m,[0m[34m'[0m[34m'[0m[34m'[0m[34m.[0m[34m.[0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m  [37m [0m[37m [0m[37m [0m
[37m [0m[37m [0m   [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[34m.[0m[34m.[0m[34m.[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m.[0m[34m.[0m[34m.[0m[34m.[0m[34m.[0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m
[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m    [37m [0m [37m [0m   [37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m.[0m[34m.[0m[34m.[0m[34m.[0m[34m'[0m[34m.[0m[34m.[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m'[0m[34m.[0m[34m.[0m[34m.[0m[34m.[0m[34m.[0m[34m.[0m[34m.[0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m[37m [0m

[36mTry out + Suite:
- Go Create > Landing Page > give it a title > Save
- Drag a heading block from the sidebar to the page body and place it in a drop zone
- Click the pencil icon > click the heading block > change the heading
"
fi
# Clean up after ourselves.
rm install.sh
