<?php

namespace Drupal\pm_carousel\Plugin\Field\FieldFormatter;

use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\EntityReferenceEntityFormatter;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Exception\ParseException;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the 'pm_carousel_container' formatter.
 */
#[FieldFormatter(
  id: 'pm_carousel_container',
  label: new TranslatableMarkup('PM Carousel container'),
  field_types: [
    'entity_reference',
  ],
)]
class PmCarouselContainerFormatter extends EntityReferenceEntityFormatter {

  /**
   * The EntityField Manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected EntityFieldManagerInterface $entityFieldManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->entityFieldManager = $container->get('entity_field.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    // Get the parent class elements.
    $elements = parent::viewElements($items, $langcode);

    $field_name = $items->getFieldDefinition()->getName();
    $entity = $items->getEntity();
    $type = $entity->getEntityTypeId();
    $bundle = $entity->bundle();
    $settings_key = $type . '-' . $bundle . '-' . $this->viewMode . '-' . $field_name;

    // Fetch the settings for the formatter.
    $formatter_settings = $this->getSettings();
    $settings_yaml = $formatter_settings['settings'];
    $settings = empty($settings_yaml) ? [] : Yaml::parse($settings_yaml);

    // Add the element properties.
    $rendered_content = [
      '#theme' => 'pm_carousel_container_formatter',
      '#content' => $elements,
      '#settings_key' => $settings_key,
      '#show_play_stop_buttons' => (bool) $formatter_settings['show_play_stop_buttons'],
      '#show_pagination_buttons' => (bool) $formatter_settings['show_pagination_buttons'],
      '#show_prev_next_buttons' => (bool) $formatter_settings['show_prev_next_buttons'],
      '#attached' => [
        'library' => ['pm_carousel/pm-carousel'],
        'drupalSettings' => [
          'pmCarousel' => [
            $settings_key => $settings,
          ],
        ],
      ],
    ];

    return [$rendered_content];
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    $field_storage_definition = $field_definition->getFieldStorageDefinition();
    return $field_storage_definition->isMultiple();
  }

  /**
   * Define the default settings for the formatter.
   */
  public static function defaultSettings() {
    return [
      'settings' => '',
      'show_play_stop_buttons' => TRUE,
      'show_pagination_buttons' => TRUE,
      'show_prev_next_buttons' => TRUE,
    ] + parent::defaultSettings();
  }

  /**
   * Define the settings form.
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    // Initialize the form array with parent's settings form.
    $form = parent::settingsForm($form, $form_state);

    $form['settings'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Carousel settings'),
      '#default_value' => $this->getSetting('settings'),
      '#description' => $this->t('Enter the PM Carousel settings in YAML format.'),
      '#rows' => 10,
      '#element_validate' => [[get_class($this), 'validateYamlSettings']],
    ];
    $form['show_play_stop_buttons'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show Play/Stop buttons'),
      '#description' => $this->t('Whether to display the Play and Stop buttons for controlling autoplay.'),
      '#default_value' => $this->getSetting('show_play_stop_buttons'),
    ];
    $form['show_pagination_buttons'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show pagination buttons'),
      '#description' => $this->t('Whether to display the pagination buttons for navigating through slides.'),
      '#default_value' => $this->getSetting('show_pagination_buttons'),
    ];
    $form['show_prev_next_buttons'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show previous/next buttons'),
      '#description' => $this->t('Whether to display the Previous and Next buttons for navigating through slides.'),
      '#default_value' => $this->getSetting('show_prev_next_buttons'),
    ];

    return $form;
  }

  /**
   * Validate the Yaml settings.
   */
  public static function validateYamlSettings(array &$element, FormStateInterface $form_state, array &$complete_form) {
    // Use Symfony YAML component for validation.
    try {
      $element_path = $element['#parents'];
      $yaml_input = $form_state->getValue($element_path);
      // No check if yaml string is empty.
      if (empty($yaml_input)) {
        return;
      }
      // If it's invalid, it will throw an exception.
      $parsed_yaml = Yaml::parse($yaml_input);
      // Check if the parsed YAML is null or empty.
      if (empty($parsed_yaml)) {
        $form_state->setErrorByName($element_path, t('The YAML input is empty.'));
      }
    }
    catch (ParseException $e) {
      // Catch the parse exception and set the form error.
      $form_state->setErrorByName($element_path, t('The YAML provided is invalid: @message', [
        '@message' => $e->getMessage(),
      ]));
    }
  }

  /**
   * Define the summary of the settings.
   */
  public function settingsSummary() {
    // Retrieve the summary from the parent class.
    $summary = parent::settingsSummary();

    $formatter_settings = $this->getSettings();
    $settings_yaml = $formatter_settings['settings'];

    if (!empty($settings_yaml)) {
      $carousel_settings = Yaml::parse($settings_yaml);
      $default_settings = $carousel_settings['default'] ?? [];
      if (!empty($default_settings)) {
        if (isset($default_settings['loop'])) {
          $summary[] = $this->t('Loop: @loop', ['@loop' => $default_settings['loop'] ? $this->t('Enabled') : $this->t('Disabled')]);
        }
        if (isset($default_settings['group'])) {
          $summary[] = $this->t('Group: @group', ['@group' => $default_settings['group']]);
        }
        if (isset($default_settings['spaceAround'])) {
          $summary[] = $this->t('Space around: @space%', ['@space' => $default_settings['spaceAround']]);
        }
        if (isset($default_settings['noStartSpace'])) {
          $summary[] = $this->t('No start space: @noStartSpace', ['@noStartSpace' => $default_settings['noStartSpace'] ? $this->t('Yes') : $this->t('No')]);
        }
        if (isset($default_settings['autoplay'])) {
          $summary[] = $this->t('Autoplay: @autoplay ms', ['@autoplay' => $default_settings['autoplay']]);
        }
        if (isset($default_settings['disable'])) {
          if ($default_settings['disable'] === 'auto') {
            $disable_label = $this->t('Automatic');
          }
          else {
            $disable_label = $default_settings['disable'] ? $this->t('Yes') : $this->t('No');
          }
          $summary[] = $this->t('Disable: @disable', ['@disable' => $disable_label]);
        }
        if (isset($default_settings['fullScroll'])) {
          $summary[] = $this->t('Full scroll: @fullScroll', ['@fullScroll' => $default_settings['fullScroll'] ? $this->t('Yes') : $this->t('No')]);
        }
      }
    }

    $summary[] = $this->t('Show Play/Stop buttons: @show_play_stop_buttons',
      ['@show_play_stop_buttons' => $formatter_settings['show_play_stop_buttons'] ? $this->t('Yes') : $this->t('No')]);
    $summary[] = $this->t('Show pagination buttons: @show_pagination_buttons',
      ['@show_pagination_buttons' => $formatter_settings['show_pagination_buttons'] ? $this->t('Yes') : $this->t('No')]);
    $summary[] = $this->t('Show previous/next buttons: @show_prev_next_buttons',
      ['@show_prev_next_buttons' => $formatter_settings['show_prev_next_buttons'] ? $this->t('Yes') : $this->t('No')]);

    return $summary;
  }

}
