<?php

namespace Drupal\pm_carousel_views\Plugin\views\style;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\views\Attribute\ViewsStyle;
use Drupal\views\Plugin\views\style\StylePluginBase;
use Symfony\Component\Yaml\Exception\ParseException;
use Symfony\Component\Yaml\Yaml;

/**
 * Defines a PM Carousel views style plugin.
 *
 * @ingroup views_style_plugins
 */
#[ViewsStyle(
  id: "pm_carousel",
  title: new TranslatableMarkup("PM Carousel"),
  help: new TranslatableMarkup("Renders view items as PM carousel slides."),
  theme: "pm_carousel_view_unformatted",
  display_types: ["normal"],
)]
class PmCarouselViewStyle extends StylePluginBase {

  /**
   * Does the style plugin allow row classes.
   *
   * @var bool
   */
  protected $usesRowClass = TRUE;

  /**
   * Indicates whether this style uses a row plugin.
   *
   * @var bool
   */
  protected $usesRowPlugin = TRUE;

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['settings'] = ['default' => ''];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);
    $form['settings'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Carousel settings'),
      '#default_value' => $this->options['settings'],
      '#description' => $this->t('Enter the PM Carousel settings in YAML format.'),
      '#rows' => 10,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state) {
    parent::submitOptionsForm($form, $form_state);
    // Use Symfony YAML component for validation.
    try {
      $yaml_input = $form_state->getValue(['style_options', 'settings']);
      // No check if yaml string is empty.
      if (empty($yaml_input)) {
        return;
      }
      // If it's invalid, it will throw an exception.
      $parsed_yaml = Yaml::parse($yaml_input);
      // Check if the parsed YAML is null or empty.
      if (empty($parsed_yaml)) {
        $form_state->setErrorByName('settings', $this->t('The YAML input is empty.'));
      }
      $this->options['settings'] = $yaml_input;
    }
    catch (ParseException $e) {
      // Catch the parse exception and set the form error.
      $form_state->setErrorByName('settings', $this->t('The YAML provided is invalid: @message', [
        '@message' => $e->getMessage(),
      ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function render() {

    $output = parent::render();

    $settings_key = $this->view->id() . '__' . $this->view->current_display;

    foreach ($output as $index => &$set) {
      $output[$index]['#settings_key'] = $settings_key;
    }

    $settings_yaml = $this->options['settings'];
    $settings = empty($settings_yaml) ? [] : Yaml::parse($settings_yaml);

    $output['#attached'] = [
      'library' => [
        'pm_carousel/pm-carousel',
      ],
      'drupalSettings' => [
        'pmCarousel' => [
          $settings_key => $settings,
        ],
      ],
    ];

    return $output;
  }

}
