# Popup Entity

A powerful Drupal module that provides configurable popup entities with comprehensive component management, weight-based field ordering, and support for webforms, blocks, custom text (rich text), custom HTML, and any created paragraph types in the system.

## Features

- **Advanced Component Management**: Add and manage multiple types of content in your popup entities
- **Webform Integration**: Embed one or more Webform module forms
- **Block Integration**: Add any site blocks to popup content
- **Custom Text Support**: Include custom text with rich text formatting capabilities
- **Custom HTML Support**: Add custom HTML content for advanced formatting
- **Paragraph Types Support**: Include any paragraph types created in your system (if Paragraphs module is available)
- **Weight-Based Ordering**: Manage component order using weight fields for precise control
- **Customizable Settings**: Control animations, dimensions, behaviors for each popup
- **Responsive Design**: Mobile-friendly popups that adapt to screen size
- **Accessibility**: Full keyboard navigation and screen reader support
- **Multiple Triggers**: Support for any CSS selector as popup trigger
- **PHP 8.3+ Optimized**: Compatible with PHP 8.3 and 8.4

## Requirements

- Drupal 10.0+ or Drupal 11.0+
- PHP 8.1+ (optimized for 8.3+)
- **Optional but recommended:**
  - Webform module (for form functionality)
  - Paragraphs module (for paragraph type components)
  - Entity Reference Revisions module (required if using Paragraphs)
  - Media module (for media components)

## Installation

1. Download and install the module using Composer:
   ```bash
   composer require drupal/popup_entity
   ```

2. Install optional dependencies for full functionality:
   ```bash
   composer require drupal/webform drupal/paragraphs drupal/entity_reference_revisions
   ```

3. Enable the module and its dependencies:
   ```bash
   drush en popup_entity webform paragraphs entity_reference_revisions media
   ```

4. Configure your popup entities at:
   `Administration » Configuration » User Interface » Popup Entity`

## Configuration

### Creating an Advanced Popup Entity

1. Go to `Administration » Configuration » User Interface » Popup Entity » Manage`
2. Click "Add Popup Entity"
3. Configure the basic settings:
   - **Label**: Administrative label for the popup
   - **Trigger Selector**: CSS selector for elements that open the popup (e.g., `#open-signup`, `.popup-trigger`)
   - **Popup Title**: Title displayed in the popup
   - **Popup Description**: Optional description text

### Adding Components

The component system allows you to add multiple types of content:

#### Webforms
- Select from available webforms on your site
- Add multiple webforms to a single popup
- Each webform maintains its own submission handling

#### Blocks
- Add any active block from your site
- Includes custom blocks, views blocks, and system blocks
- Blocks render with their original configuration

#### Custom Text (Rich Text)
- Add custom text content with rich text formatting
- Supports HTML formatting, links, and basic styling
- Perfect for introductory text, instructions, or calls-to-action

#### Custom HTML
- Include custom HTML content for advanced formatting
- Full HTML support for complex layouts
- Ideal for custom styling and advanced content structures

#### Paragraph Types
- Choose from any paragraph types created in your system
- Automatically displays available paragraph types when Paragraphs module is enabled
- Create paragraph content with all field types supported
- Select view mode (default, full, preview)
- Perfect for reusable content components and structured data

### Component Management

- **Weight-Based Ordering**: Use weight fields to control the order of components in your popup
- **Flexible Ordering**: Components are sorted by weight value (lower values appear first)
- **Real-time Preview**: Preview your popup configuration before saving
- **AJAX Interface**: Add and remove components without page reloads

### Popup Settings

Customize popup behavior:
- **Dimensions**: Set width and height (supports px, %, vw, vh)
- **Animation**: Choose from fade, slide, zoom effects
- **Behavior**: Control overlay, close button, escape key, click-outside-close
- **Auto Close**: Optionally auto-close after a specified delay

## Usage Examples

### Basic Usage with Multiple Components

```html
<!-- HTML trigger element -->
<button id="comprehensive-signup">Complete Registration</button>
```

**Popup Configuration:**
- Trigger Selector: `#comprehensive-signup`
- Components (ordered by weight):
  1. Custom Text (welcome message with rich text formatting) - Weight: 0
  2. Webform (registration form) - Weight: 10
  3. Custom HTML (promotional banner with custom styling) - Weight: 20
  4. Block (related links) - Weight: 30
  5. Paragraph Type (terms and conditions content) - Weight: 40

### Advanced Multi-Step Process

```html
<!-- Multi-step signup trigger -->
<a href="#" class="multi-step-trigger" data-step="1">Start Application</a>
```

**Multiple Popup Configuration:**
- Step 1: Custom text introduction (weight: 0) + basic info webform (weight: 10)
- Step 2: Paragraph type - detailed instructions (weight: 0) + complex webform (weight: 10)
- Step 3: Custom HTML - confirmation message (weight: 0) + final block (weight: 10)

### Content Showcase Popup

```html
<!-- Product showcase -->
<button class="product-showcase" data-product="premium">View Premium Package</button>
```

**Components:**
1. Custom HTML (product header with custom styling) - Weight: 0
2. Paragraph Type (feature list with structured fields) - Weight: 10
3. Custom Text (pricing information with rich text) - Weight: 20
4. Webform (contact form) - Weight: 30
5. Block (testimonials) - Weight: 40

## Programmatic Usage

### PHP API

```php
// Load popup entity manager
$popup_manager = \Drupal::service('popup_entity.popup_manager');

// Get all active popup entities
$popup_entities = $popup_manager->getActivePopupEntities();

// Create a new popup entity programmatically
$popup_entity = \Drupal\popup_entity\Entity\PopupEntity::create([
  'id' => 'my_custom_popup',
  'label' => 'My Custom Popup',
  'trigger_selector' => '#my-trigger',
  'popup_title' => 'Welcome!',
]);

// Add components with specific weights for ordering
$popup_entity->addComponent('webform', 'contact', ['default_data' => ['name' => 'John']], 10);
$popup_entity->addComponent('custom_text', '', ['content' => '<p><strong>Welcome!</strong> Please fill out the form below.</p>'], 5);
$popup_entity->addComponent('paragraph_type', 'welcome_content', [], 15);
$popup_entity->addComponent('custom_html', '', ['content' => '<div class="banner">Special Offer!</div>'], 0);
$popup_entity->addComponent('block', 'social_links', [], 20);

$popup_entity->save();

// Render popup content
$content = $popup_manager->renderPopupContent($popup_entity);
```

### JavaScript API

```javascript
// Open a popup programmatically
Drupal.popupEntity.openEnhanced('popup_id', {
  title: 'Dynamic Popup',
  content: '<div>Custom content</div>',
  settings: {
    width: '800px',
    animation: 'slideDown'
  }
});

// Close a popup
Drupal.popupEntity.close('popup_id');

// Listen for popup events
document.addEventListener('popupEntityOpened', function(e) {
  console.log('Popup opened:', e.detail.popupId);
});
```

## Available Component Types

### 1. Webforms
- **Purpose**: Embed interactive forms for data collection
- **Configuration**: Select from available webforms, set default data, assign weight for ordering
- **Use Cases**: Contact forms, registration, surveys, feedback

### 2. Blocks
- **Purpose**: Include any configured block from your site
- **Configuration**: Choose from active blocks, maintains original settings, set weight for ordering
- **Use Cases**: Navigation menus, recent content, custom blocks, views

### 3. Custom Text (Rich Text)
- **Purpose**: Add formatted text content with WYSIWYG editing
- **Configuration**: Rich text editor with formatting options, weight field for positioning
- **Use Cases**: Welcome messages, instructions, formatted announcements

### 4. Custom HTML
- **Purpose**: Include custom HTML for advanced layouts and styling
- **Configuration**: Raw HTML input with full HTML support, weight-based ordering
- **Use Cases**: Custom banners, complex layouts, embedded widgets

### 5. Paragraph Types (when available)
- **Purpose**: Use structured content created with Paragraphs module
- **Configuration**: Select paragraph type, fill in fields, choose view mode, set weight
- **Use Cases**: Structured content, reusable components, complex field groups
- **Field Types Supported**: Text, images, links, numbers, dates, references, and more
- **Ordering**: Use weight field to control position in popup (lower weights appear first)

## Theming

### CSS Classes

- `.popup-entity-wrapper`: Main wrapper
- `.popup-overlay`: Background overlay
- `.popup-container`: Popup container
- `.popup-content`: Content area
- `.popup-component`: Individual component wrapper
- `.popup-component-webform`: Webform components
- `.popup-component-custom-text`: Custom text components
- `.popup-component-custom-html`: Custom HTML components
- `.popup-component-paragraph-type`: Paragraph type components
- `.popup-component-block`: Block components

### Component-Specific Classes

```css
/* Custom Text styling */
.popup-component-custom-text {
  padding: 15px 0;
  line-height: 1.6;
}

/* Custom HTML styling */
.popup-component-custom-html {
  margin: 10px 0;
}

/* Paragraph Type styling */
.popup-component-paragraph-type {
  border-left: 4px solid #28a745;
  padding-left: 15px;
  margin: 15px 0;
}

/* Webform styling */
.popup-component-webform .webform-submission-form {
  margin: 0;
  padding: 20px;
}

/* Block styling */
.popup-component-block {
  background: #f8f9fa;
  padding: 15px;
  border-radius: 4px;
  margin: 10px 0;
}
```

### Custom Themes

Override default styles by setting `include_default_css: false` in settings and providing your own CSS.

## API Reference

### Hooks

```php
/**
 * Alter popup entity configuration before rendering.
 */
function hook_popup_entity_alter(&$popup_entity) {
  if ($popup_entity->id() === 'my_popup') {
    // Modify popup configuration
    $popup_entity->setPopupTitle('Modified Title');
  }
}

/**
 * Alter rendered popup content.
 */
function hook_popup_entity_content_alter(&$content, $popup_entity) {
  $content['custom_footer'] = [
    '#markup' => '<div class="popup-footer">Custom footer content</div>',
    '#weight' => 100,
  ];
}

/**
 * React to popup entity component rendering.
 */
function hook_popup_entity_component_alter(&$component_render, $component, $popup_entity) {
  if ($component['type'] === 'custom_text') {
    // Modify custom text component rendering
    $component_render['#prefix'] = '<div class="custom-text-wrapper">';
    $component_render['#suffix'] = '</div>';
  }
}
```

### Services

- `popup_entity.popup_manager`: Main service for popup entity management and rendering

## Development

### Contributing

1. Fork the repository
2. Create a feature branch
3. Make your changes following Drupal coding standards
4. Add tests for new functionality
5. Submit a pull request

### Testing

```bash
# Run PHPUnit tests
./vendor/bin/phpunit modules/contrib/popup_entity/tests/

# Run coding standards check
./vendor/bin/phpcs --standard=Drupal modules/contrib/popup_entity/

# Run static analysis (requires PHPStan)
./vendor/bin/phpstan analyze modules/contrib/popup_entity/
```

### Debugging

Enable debug mode in popup entity settings to get detailed logging:

```php
// In settings.php
$config['popup_entity.settings']['debug_mode'] = TRUE;
```

## Migration from Previous Versions

The module includes automatic migration from older versions. Run database updates after upgrading:

```bash
drush updb
```

## Performance Considerations

- Components are rendered on-demand when popups are triggered
- Paragraph types are loaded with appropriate view modes to optimize performance  
- Block caching is respected for block components
- Consider enabling page caching for sites with many popup entities

## Browser Support

- Modern browsers (Chrome 90+, Firefox 88+, Safari 14+, Edge 90+)
- Progressive enhancement for older browsers
- Responsive design for mobile devices
- Accessibility features for screen readers

## License

This project is licensed under the GPL-2.0+ License.

## Support

- Issue Queue: https://www.drupal.org/project/issues/popup_entity
- Documentation: https://www.drupal.org/docs/contributed-modules/popup-entity
- Community Support: https://drupal.org/slack (Channel: #popup-entity)

## Changelog

### 2.0.0
- Added component-based architecture
- Support for custom text (rich text) and custom HTML
- Support for paragraph types with field creation
- Support for blocks and webforms
- Weight-based component ordering system
- PHP 8.3+ optimization
- Improved accessibility
- Migration path from 1.x versions