<?php

namespace Drupal\popup\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Extension\ModuleExtensionList;

/**
 * Provides a 'Popup' block.
 *
 * @Block(
 * id = "popup",
 * admin_label = @Translation("Popup"),
 * category = @Translation("Popup")
 * )
 */
class PopupBlock extends BlockBase implements ContainerFactoryPluginInterface {

  protected $moduleExtensionList;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, ModuleExtensionList $moduleExtensionList) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->moduleExtensionList = $moduleExtensionList;
  }

  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('extension.list.module')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);
    $config = $this->getConfiguration();

    $form['top_text_config'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Popup Top Text'),
      '#description' => $this->t('The text that will appear at the top of the popup.'),
      '#default_value' => $config['top_text_config'] ?? $this->t('Hello!'),
    ];

    $form['bottom_text_config'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Popup Bellow Image Text'),
      '#description' => $this->t('The text that will appear at the bottom of the popup.'),
      '#default_value' => $config['bottom_text_config'] ?? $this->t('Welcome to our website.'),
    ];

    $form['image_url_config'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Popup Main Image URL'),
      '#description' => $this->t('URL of the main image.'),
      '#default_value' => $config['image_url_config'] ?? $this->getDefaultImagePath(),
    ];

    $form['bottom_image_url_config'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Popup Bottom Image URL'),
      '#description' => $this->t('URL of the bottom image.'),
      '#default_value' => $config['bottom_image_url_config'] ?? $this->getDefaultImagePath(),
    ];
    $form['popup_background_image'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Popup Background Image URL'),
      '#description' => $this->t('URL of the pop up background image.'),
      '#default_value' => $config['popup_background_image'] ?? $this->getDefaultImagePath(),
    ];
    $form['popup_background_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Popup Background Color'),
      '#description' => $this->t('Select the background color for the popup window.'),
      '#default_value' => $config['popup_background_color'] ?? '#441a1aff',
    ];

    return $form;
  }

  /**
   * Get default image path.
   */
  protected function getDefaultImagePath() {

    $path = $this->moduleExtensionList->getPath('popup');
    return '/' . $path . '/images/default.jpg';
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    $this->configuration['top_text_config'] = $form_state->getValue('top_text_config');
    $this->configuration['bottom_text_config'] = $form_state->getValue('bottom_text_config');
    $this->configuration['image_url_config'] = $form_state->getValue('image_url_config');
    $this->configuration['bottom_image_url_config'] = $form_state->getValue('bottom_image_url_config');
    $this->configuration['popup_background_color'] = $form_state->getValue('popup_background_color');
    $this->configuration['popup_background_image'] = $form_state->getValue('popup_background_image');
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $config = $this->getConfiguration();
    
    $top_text = $config['top_text_config'] ?? $this->t('Happy Holidays!');
    $bottom_text = $config['bottom_text_config'] ?? $this->t('Welcome to our website.');
    $image_path = $config['image_url_config'] ?? $this->getDefaultImagePath();
    $bottom_image_path = $config['bottom_image_url_config'] ?? $this->getDefaultImagePath();
    $bg_color = $config['popup_background_color'] ?? $this->defaultConfiguration()['popup_background_color'];
    $bg_image = $config['popup_background_image'] ?? $this->getDefaultImagePath();

    return [
      '#theme' => 'popup_block',
      '#top_text' => $top_text,
      '#bottom_text' => $bottom_text,
      '#image_url' => $image_path,
      '#bottom_image_url' => $bottom_image_path,
      '#background_color' => $bg_color,
      '#background_image' => $bg_image,
      '#attached' => [
        'library' => [
          'popup/popup',
        ],
      ],
    ];
  }
}