<?php

namespace Drupal\popup\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\node\Entity\NodeType;
use Drupal\node\NodeInterface;

/**
 * Provides a 'Popup' block.
 *
 * @Block(
 *   id = "popup",
 *   admin_label = @Translation("Popup"),
 *   category = @Translation("Popup")
 * )
 */
class PopupBlock extends BlockBase implements ContainerFactoryPluginInterface {

  protected ModuleExtensionList $moduleExtensionList;

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    ModuleExtensionList $moduleExtensionList
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->moduleExtensionList = $moduleExtensionList;
  }

  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition
  ) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('extension.list.module')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);
    $config = $this->getConfiguration();

    $form['top_text_config'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Popup Top Text'),
      '#default_value' => $config['top_text_config'] ?? 'Hello!',
    ];

    $form['bottom_text_config'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Popup Bottom Text'),
      '#default_value' => $config['bottom_text_config'] ?? 'Welcome to our website.',
    ];

    $form['image_url_config'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Popup Main Image URL'),
      '#default_value' => $config['image_url_config'] ?? $this->getDefaultImagePath(),
    ];

    $form['bottom_image_url_config'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Popup Bottom Image URL'),
      '#default_value' => $config['bottom_image_url_config'] ?? $this->getDefaultImagePath(),
    ];

    $form['popup_background_image'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Popup Background Image URL'),
      '#default_value' => $config['popup_background_image'] ?? $this->getDefaultImagePath(),
    ];

    $form['popup_background_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Popup Background Color'),
      '#default_value' => $config['popup_background_color'] ?? '#441a1a',
    ];

    $node_types = NodeType::loadMultiple();
    $options = [];

    foreach ($node_types as $type) {
      $options[$type->id()] = $type->label();
    }

    $form['allowed_node_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Show popup only on these content types'),
      '#description' => $this->t('Popup will only appear on selected node types.'),
      '#options' => $options,
      '#default_value' => $config['allowed_node_types'] ?? [],
    ];

    return $form;
  }

  protected function getDefaultImagePath() {
    $path = $this->moduleExtensionList->getPath('popup');
    return '/' . $path . '/images/default.jpg';
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    $this->configuration['top_text_config'] = $form_state->getValue('top_text_config');
    $this->configuration['bottom_text_config'] = $form_state->getValue('bottom_text_config');
    $this->configuration['image_url_config'] = $form_state->getValue('image_url_config');
    $this->configuration['bottom_image_url_config'] = $form_state->getValue('bottom_image_url_config');
    $this->configuration['popup_background_color'] = $form_state->getValue('popup_background_color');
    $this->configuration['popup_background_image'] = $form_state->getValue('popup_background_image');
    $this->configuration['allowed_node_types'] = array_filter(
      $form_state->getValue('allowed_node_types') ?? []
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $config = $this->getConfiguration();
    $allowed_node_types = array_filter($config['allowed_node_types'] ?? []);

    if (empty($allowed_node_types)) {
      return [];
    }

    $node = \Drupal::routeMatch()->getParameter('node');
    if (!$node instanceof NodeInterface) {
      return [];
    }

    if (!in_array($node->bundle(), $allowed_node_types, TRUE)) {
      return [];
    }

    return [
      '#theme' => 'popup_block',
      '#top_text' => $config['top_text_config'],
      '#bottom_text' => $config['bottom_text_config'],
      '#image_url' => $config['image_url_config'],
      '#bottom_image_url' => $config['bottom_image_url_config'],
      '#background_color' => $config['popup_background_color'],
      '#background_image' => $config['popup_background_image'],
      '#attached' => [
        'library' => [
          'popup/popup',
        ],
      ],
      '#cache' => [
        'contexts' => [
          'route',
        ],
      ],
    ];
  }

}
