<?php

declare(strict_types=1);

namespace Drupal\postal_mail\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Controller\ControllerBase;
use Drupal\postal_mail\Event\WebhookEvent;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for handling webhooks.
 */
class WebhookController extends ControllerBase {

  protected ImmutableConfig $config;
  protected EventDispatcherInterface $eventDispatcher;

  /**
   * WebhookController constructor.
   */
  final public function __construct(EventDispatcherInterface $event_dispatcher, ConfigFactoryInterface $config_factory) {
    $this->eventDispatcher = $event_dispatcher;
    $this->config = $config_factory->get('postal_mail.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('event_dispatcher'),
      $container->get('config.factory')
    );
  }

  /**
   * Checks access to webhook endpoint.
   */
  public function access(Request $request): AccessResultInterface {
    $secret = $request->get('secret', FALSE);
    if ($secret == $this->config->get('webhook_secret')) {
      return AccessResult::allowed();
    }
    return AccessResult::forbidden();
  }

  /**
   * Handle the webhook.
   */
  public function webhook(Request $request): Response {
    $content = $request->getContent();
    if (!empty($content)) {
      $payload = json_decode($content, TRUE);
      $event = new WebhookEvent($payload);
      $this->eventDispatcher->dispatch($event, WebhookEvent::EVENT_NAME);
    }
    return new Response('', 200);
  }

}
