<?php

declare(strict_types=1);

namespace Drupal\postal_mail\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Settings form for Postal Mail Delivery Platform Integration.
 */
class SettingsForm extends ConfigFormBase {

  protected ImmutableConfig $config;

  /**
   * SettingsForm constructor.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->config = $config_factory->get('postal_mail.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'postal_mail_settings';
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Exception
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['host'] = [
      '#title' => $this->t('Host'),
      '#type' => 'textfield',
      '#default_value' => $this->config->get('host'),
      '#required' => TRUE,
    ];

    $form['api_key'] = [
      '#title' => $this->t('API Key'),
      '#type' => 'textfield',
      '#default_value' => $this->config->get('api_key'),
      '#required' => TRUE,
    ];

    $form['webhook_secret'] = [
      '#title' => $this->t('Webhook secret'),
      '#type' => 'textfield',
      '#default_value' => $this->config->get('webhook_secret'),
      '#description' => $this->t('The webhook secret is used to verify incoming webhook requests. It should be a random string that is kept secret.'),
      '#required' => TRUE,
    ];

    if (!empty($this->config->get('webhook_secret'))) {
      $url = Url::fromRoute('postal_mail.webhook', [], ['absolute' => TRUE, 'query' => ['secret' => $this->config->get('webhook_secret')]])->toString();
      $form['webhook_uri'] = [
        '#markup' => '<p>' . $this->t('<strong>Webhook URL:</strong> <code>@url</code>', ['@url' => $url]) . '</p>',
      ];
    }

    $form['from_mail'] = [
      '#title' => $this->t('From mail'),
      '#type' => 'textfield',
      '#default_value' => $this->config->get('from_mail'),
      '#required' => TRUE,
    ];

    $form['from_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('From name'),
      '#default_value' => $this->config->get('from_name'),
    ];

    $form['use_queue'] = [
      '#title' => $this->t('Queue outgoing messages'),
      '#type' => 'checkbox',
      '#description' => $this->t('When set, emails will not be immediately sent. Instead, they will be placed in a queue and sent when cron is triggered.'),
      '#default_value' => $this->config->get('use_queue'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->configFactory()->getEditable('postal_mail.settings')
      ->set('host', $form_state->getValue('host'))
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('webhook_secret', $form_state->getValue('webhook_secret'))
      ->set('from_mail', $form_state->getValue('from_mail'))
      ->set('from_name', $form_state->getValue('from_name'))
      ->set('use_queue', $form_state->getValue('use_queue'))
      ->save();
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['postal_mail.settings'];
  }

}
