<?php

declare(strict_types=1);

namespace Drupal\postal_mail\Form;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Url;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for sending test mail.
 *
 * @ingroup mailchimp_transactional
 */
class TestMailForm extends ConfirmFormBase {

  protected MailManagerInterface $mailManager;
  protected FileSystemInterface $fileSystem;
  protected LanguageManagerInterface $languageManager;

  /**
   * TestMailForm constructor.
   */
  final public function __construct(MailManagerInterface $mail_manager, FileSystemInterface $file_system, LanguageManagerInterface $language_manager) {
    $this->mailManager = $mail_manager;
    $this->fileSystem = $file_system;
    $this->languageManager = $language_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.mail'),
      $container->get('file_system'),
      $container->get('language_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'postal_mail_test_email';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Postal Mail Delivery Platform Test Email');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->t('This action will send a test email through Postal Mail Delivery Platform.');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('postal_mail.test');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Send test email');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);

    $form['to'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Send to'),
      '#default_value' => $this->configFactory()->get('system.site')->get('mail'),
      '#description' => $this->t('Multiple addresses allowed with comma separation, including <code>name &lt;email@example.com&gt;</code> formatting.'),
      '#required' => TRUE,
    ];

    $form['body'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Message Content'),
      '#default_value' => '<h2>Congratulations, it works!</h2><p>This message was sent from a Drupal site using <a href="https://www.drupal.org/project/postal_mail">Postal Mail Delivery Platform</a> to send email.</p>',

    ];

    $form['include_attachment'] = [
      '#title' => $this->t('Include attachment'),
      '#type' => 'checkbox',
      '#description' => $this->t('If checked, the Drupal icon will be included as an attachment.'),
      '#default_value' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $langcode = $this->languageManager->getCurrentLanguage()->getId();

    $body = $form_state->getValue('body');
    $params['body'] = $body['value'];
    $params['subject'] = $this->t('Drupal Postal Mail Delivery Platform Test Email');

    if ($form_state->getValue('include_attachment')) {
      $file = $this->fileSystem->realpath('core/themes/stark/logo.svg');
      $params['attachments'][] = [
        'filename' => 'drupal-icon.svg',
        'filecontent' => file_get_contents($file),
        'filemime' => 'image/svg+xml',
      ];
      $params['body'] .= PHP_EOL . '<p>The Drupal icon is included as an attachment to test the attachment functionality.</p>';
    }

    $result = $this->mailManager->mail('postal_mail', 'test', $form_state->getValue('to'), $langcode, $params, NULL, TRUE);
    if ($result['result']) {
      $this->messenger()->addMessage($this->t('The test email has been sent.'));
    }
    else {
      $this->messenger()->addError($this->t('There was a problem sending the test email.'));
    }
  }

}
