<?php

declare(strict_types=1);

namespace Drupal\postal_mail\Plugin\Mail;

use Donchev\EmailExtractor\EmailExtractor;
use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Mail\MailInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Mail\MailFormatHelper;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\postal_mail\PostalAPI;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Mime\MimeTypeGuesserInterface;

/**
 * Allow Drupal mailsystem to use Postal Mail Delivery Platform when sending emails.
 *
 * @Mail(
 *   id = "postal_mail",
 *   label = @Translation("Postal Mail Delivery Platform"),
 *   description = @Translation("Sends the message through Postal Mail Delivery Platform.")
 * )
 */
class PostalMail extends PluginBase implements MailInterface, ContainerFactoryPluginInterface {

  protected ImmutableConfig $config;
  protected LoggerChannelInterface $log;
  protected MimeTypeGuesserInterface $mimeTypeGuesser;
  protected QueueInterface $queue;
  protected PostalAPI $postalAPI;
  protected ModuleHandlerInterface $moduleHandler;
  protected FileSystemInterface $fileSystem;
  protected RendererInterface $renderer;

  /**
   * PostalMail constructor.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_channel_factory,
    MimeTypeGuesserInterface $mime_type_guesser,
    QueueFactory $queue_factory,
    PostalAPI $postal_api,
    ModuleHandlerInterface $module_handler,
    FileSystemInterface $file_system,
    RendererInterface $renderer
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->config = $config_factory->get('postal_mail.settings');
    $this->log = $logger_channel_factory->get('postal_mail');
    $this->mimeTypeGuesser = $mime_type_guesser;
    $this->queue = $queue_factory->get('postal_mail_queue', TRUE);
    $this->postalAPI = $postal_api;
    $this->moduleHandler = $module_handler;
    $this->fileSystem = $file_system;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('logger.factory'),
      $container->get('file.mime_type.guesser'),
      $container->get('queue'),
      $container->get('postal_mail.api'),
      $container->get('module_handler'),
      $container->get('file_system'),
      $container->get('renderer')
    );
  }

  /**
   * Render the email message, both HTML and plain text versions.
   *
   * @param array $message
   *   A message array, as described in hook_mail_alter().
   *
   * @return array
   *   The formatted $message.
   */
  public function format(array $message): array {
    $body = $message['body'];
    if (is_array($body)) {
      $body = implode(PHP_EOL, $body);
    }

    // Get user based on email address. Notice that the 'to' value can contain
    // multiple email addresses. To deal with this we extract all values and
    // only use the first one.
    $extractor = new EmailExtractor();
    $recipients = $extractor->extract($message['to'])
      ->lower()
      ->unique()
      ->export();
    $user = user_load_by_mail(reset($recipients));

    // Assemble render array.
    $build = [
      '#theme' => 'postal_mail_message',
      '#user' => $user,
      '#module' => $message['module'],
      '#key' => $message['key'],
      '#subject' => $message['subject'],
      '#body' => $body,
      '#params' => $message['params'],
    ];

    $message['body'] = (string) $this->renderer->renderInIsolation($build);
    $message['plain_text'] = MailFormatHelper::htmlToText($message['body']);

    return $message;
  }

  /**
   * Send the email message.
   *
   * @param array $message
   *   A message array, as described in hook_mail_alter().
   *
   * @return bool
   *   TRUE if the mail was successfully accepted, otherwise FALSE.
   *
   * @see drupal_mail()
   */
  public function mail(array $message): bool {
    // Handle attachments.
    $attachments = [];
    if (!empty($message['params']['attachments'])) {
      foreach ($message['params']['attachments'] as $attachment) {
        if (isset($attachment['uri'])) {
          $attachment_path = $this->fileSystem->realpath($attachment['uri']);
          if (is_file($attachment_path)) {
            $struct = $this->getAttachmentStruct($attachment_path);
            $attachments[] = $struct;
          }
        }
        // Support attachments that are directly included without a file in the
        // filesystem.
        elseif (isset($attachment['filecontent'])) {
          $attachments[] = [
            'name' => $attachment['filename'],
            'content_type' => $attachment['filemime'],
            'data' => chunk_split(base64_encode($attachment['filecontent'])),
          ];
        }
      }
    }

    // Different modules are using different capitalization in the keys of
    // the header array i.e. Reply-to and Reply-To.
    $headers = $message['headers'] ?? [];
    $header_key_map = [];
    foreach (array_keys($headers) as $header_key) {
      $header_key_map[strtolower($header_key)] = $header_key;
    }

    // Initialize the email extractor. We use this as the 'to', 'cc', and 'bcc'
    // values are RFC 2822 compliant strings meaning they can contain
    // multiple values i.e. 'John Doe <john.doe@mail.com>, jane.doe@mail.com'.
    $extractor = new EmailExtractor();

    // Assemble recipients from the mail message and header data.
    $recipients['to'] = $extractor
      ->extract($message['to'])
      ->lower()
      ->unique()
      ->export();

    // Get 'cc' recipients if set.
    if (isset($header_key_map['cc']) && isset($headers[$header_key_map['cc']])) {
      $recipients['cc'] = $extractor
      ->extract($headers[$header_key_map['cc']])
        ->lower()
        ->unique()
        ->export();
    }

    // Get 'bcc' recipients if set.
    if (isset($header_key_map['bcc']) && isset($headers[$header_key_map['bcc']])) {
      $recipients['bcc'] = $extractor
        ->extract($headers[$header_key_map['bcc']])
        ->lower()
        ->unique()
        ->export();
    }

    // Get reply-to value, default to from_mail if not set.
    $reply_to = $this->config->get('from_mail');
    if (isset($header_key_map['reply-to']) && isset($headers[$header_key_map['reply-to']])) {
      $reply_to = $headers[$header_key_map['reply-to']];
    }

    // Assemble sender name and email.
    $name = $this->config->get('from_name');
    $from = $this->config->get('from_mail');
    if (!empty($name)) {
      $from = $name . ' <' . $from . '>';
    }

    // Unset all headers that contain values accepted as parameters by the
    // Postal API and also default headers set by Drupal that we don't need.
    unset(
      // Headers to be set by Postal.
      $headers[$header_key_map['to'] ?? NULL],
      $headers[$header_key_map['cc'] ?? NULL],
      $headers[$header_key_map['bcc'] ?? NULL],
      $headers[$header_key_map['from'] ?? NULL],
      $headers[$header_key_map['sender'] ?? NULL],
      $headers[$header_key_map['reply-to'] ?? NULL],
      // Headers set by Drupal we don't need.
      $headers[$header_key_map['content-type'] ?? NULL],
      $headers[$header_key_map['mime-version'] ?? NULL],
      $headers[$header_key_map['content-transfer-encoding'] ?? NULL],
      $headers[$header_key_map['return-path'] ?? NULL]
    );

    // Assemble message data.
    $data = [
      'to' => $recipients['to'],
      'cc' => $recipients['cc'] ?? [],
      'bcc' => $recipients['bcc'] ?? [],
      'from' => $from,
      'sender' => $from,
      'subject' => $message['subject'],
      'tag' => $message['params']['postal_mail']['tag'] ?? '',
      'reply_to' => $reply_to,
      'plain_body' => $message['plain_text'],
      'html_body' => $message['body'],
      'attachments' => $attachments,
      'headers' => $headers,
    ];

    // Allow other modules to alter the message data.
    $this->moduleHandler->alter('postal_mail', $data, $message);

    // Check if we should send the email.
    if ($message['send'] === FALSE) return FALSE;

    // Queue for processing during cron or send immediately.
    if ($this->config->get('use_queue')) {
      $this->queue->createItem($data);
      return TRUE;
    }
    else {
      return $this->postalAPI->send($data);
    }
  }

  /**
   * Return an array structure for a message attachment.
   */
  public function getAttachmentStruct(string $path): array {
    $struct = [];
    if (!@is_file($path)) {
      throw new \Exception($path . ' is not a valid file.');
    }
    $filename = basename($path);
    $file_buffer = file_get_contents($path);
    $file_buffer = chunk_split(base64_encode($file_buffer), 76, "\n");
    $mime_type = $this->mimeTypeGuesser->guessMimeType($path);

    if (!$this->isValidContentType($mime_type)) {
      throw new \Exception($mime_type . ' is not a valid content type.');
    }
    $struct['name'] = $filename;
    $struct['content_type'] = $mime_type;
    $struct['data'] = $file_buffer;
    return $struct;
  }

  /**
   * Helper to determine if an attachment is valid.
   */
  protected function isValidContentType(string $file_type): bool {
    $valid_types = [
      'image/',
      'text/',
      'application/pdf',
      'application/x-zip',
    ];
    $this->moduleHandler->alter('postal_mail_valid_attachment_types', $valid_types);
    foreach ($valid_types as $vct) {
      if (str_contains($file_type, $vct)) {
        return TRUE;
      }
    }
    return FALSE;
  }

}
