<?php

declare(strict_types=1);

namespace Drupal\postal_mail;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Postal\ApiException;
use Postal\Client;
use Postal\Send\Message as SendMessage;
use Postal\Messages\Message;

/**
 * Class PostalAPI.
 */
class PostalAPI {

  protected ImmutableConfig $config;
  protected LoggerChannelInterface $log;
  protected Client $client;

  /**
   * PostalAPI constructor.
   */
  public function __construct(ConfigFactoryInterface $configFactory, LoggerChannelFactoryInterface $loggerChannelFactory) {
    $this->config = $configFactory->get('postal_mail.settings');
    $this->log = $loggerChannelFactory->get('postal_mail');
    $this->client = new Client($this->config->get('host'), $this->config->get('api_key'));
  }

  /**
   * Send message.
   */
  public function send(array $data): bool {
    $message = new SendMessage();
    $message->to = $data['to'];
    $message->cc = $data['cc'];
    $message->bcc = $data['bcc'];
    $message->from($data['from']);
    if (!empty($data['sender']))
      $message->sender($data['sender']);
    $message->subject($data['subject']);
    $message->tag($data['tag']);
    $message->replyTo($data['reply_to']);
    if (!empty($data['plain_body']))
      $message->plainBody($data['plain_body']);
    $message->htmlBody($data['html_body']);
    $message->attachments = $data['attachments'];
    $message->headers = $data['headers'];

    try {
      $result = $this->client->send->message($message);
    }
    catch (ApiException $e) {
      $this->log->error($e->getMessage());
      return FALSE;
    }

    foreach ($result->recipients() as $email => $message) {
      $this->log->debug('Message sent to %email with id %id and token %token.',
        [
          '%email' => $email,
          '%id' => $message->id,
          '%token' => $message->token
        ]
      );
    }

    return TRUE;
  }

  /**
   * Get details about a message.
   *
   * @throws ApiException
   */
  public function details(int $id, array $expansions = []): Message {
    try {
      return $this->client->messages->details($id, $expansions);
    }
    catch (ApiException $e) {
      $this->log->error($e->getMessage());
      throw $e;
    }
  }

  /**
   * Returns an array of deliveries which have been attempted for this message.
   *
   * @throws ApiException
   */
  public function delivery(int $id): array {
    try {
      return $this->client->messages->deliveries($id);
    }
    catch (ApiException $e) {
      $this->log->error($e->getMessage());
      throw $e;
    }
  }

}
