<?php

namespace Drupal\posthog_commerce\EventSubscriber;

use Drupal\commerce_cart\Event\CartEvents;
use Drupal\posthog_php\SdkDecoratorInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Drupal\commerce_cart\Event\CartEntityAddEvent;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * The order subscriber.
 */
class CartSubscriber implements EventSubscriberInterface {

  public const POSTHOG_CART_ADD_EVENT = 'Product Added';

  /**
   * Constructs a new order subscriber object.
   *
   * @param \Drupal\posthog_php\SdkDecoratorInterface $posthogPhpSdk
   *   The PostHog PHP SDK.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerChannelFactory
   *   The logger channel factory.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(
    protected readonly SdkDecoratorInterface $posthogPhpSdk,
    protected readonly LoggerChannelFactoryInterface $loggerChannelFactory,
    protected readonly ConfigFactoryInterface $configFactory,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      CartEvents::CART_ENTITY_ADD => 'entityAdded',
    ];
  }

  /**
   * Listens and emits the posthog product added event.
   *
   * @param \Drupal\commerce_cart\Event\CartEntityAddEvent $event
   *   The event.
   */
  public function entityAdded(CartEntityAddEvent $event) {
    // Check, if the event should be tracked:
    $cartAddedEnabled = $this->configFactory->get('posthog_commerce.settings')->get('track_commerce_cart_entity_added');
    if (!$cartAddedEnabled) {
      return;
    }
    // It is never worth to have errors on the checkout, so we capsule
    // everything in a try-catch block:
    try {
      $orderItem = $event->getOrderItem();
      $quantity = $event->getQuantity();
      $cart = $event->getCart();
      $purchasableEntity = $event->getEntity();

      $order = $orderItem->getOrder();
      $customer = $order->getCustomer();

      $properties = [
        'cart_id' => $cart->id(),
        'product_id' => $purchasableEntity->id(),
        'name' => $purchasableEntity->label(),
        'sku' => $purchasableEntity?->getSku(),
        'price' => $purchasableEntity?->getPrice()?->getNumber(),
        'quantity' => $quantity,
      ];
      $this->posthogPhpSdk->captureUserEvent(self::POSTHOG_CART_ADD_EVENT, $properties, [
        'commerce_order' => $order,
        'user' => $customer,
      ]);
    }
    catch (\Exception $e) {
      if ($this->configFactory->get('posthog_php.settings')->get('enable_logging')) {
        $logger = $this->loggerChannelFactory->get('posthog_commerce');
        $logger->notice('Something went wrong, while capturing the posthog @event. Error message: @error_message', [
          '@event' => self::POSTHOG_CART_ADD_EVENT,
          '@error_message' => $e->getMessage(),
        ]);
      }
    }

  }

}
