<?php

namespace Drupal\posthog_commerce\EventSubscriber;

use Drupal\commerce_order\Event\OrderEvent;
use Drupal\commerce_order\Event\OrderEvents;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\posthog_php\SdkDecoratorInterface;
use Drupal\state_machine\Event\WorkflowTransitionEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * The order subscriber.
 */
class OrderSubscriber implements EventSubscriberInterface {

  public const POSTHOG_ORDER_COMPLETED_EVENT = 'Order Completed';

  /**
   * Constructs a new order subscriber object.
   *
   * @param \Drupal\posthog_php\SdkDecoratorInterface $posthogPhpSdk
   *   The PostHog PHP SDK.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerChannelFactory
   *   The logger channel factory.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(
    protected readonly SdkDecoratorInterface $posthogPhpSdk,
    protected readonly LoggerChannelFactoryInterface $loggerChannelFactory,
    protected readonly ConfigFactoryInterface $configFactory,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      OrderEvents::ORDER_PAID => 'orderPaid',
      'commerce_order.place.post_transition' => 'orderCompleted',
    ];
  }

  /**
   * Listens on the order paid event and emits the posthog order paid event.
   *
   * @param \Drupal\commerce_order\Event\OrderEvent $event
   *   The cart entity add event.
   */
  public function orderPaid(OrderEvent $event) {
    // Check, if the event should be tracked:
    $orderPaidEnabled = $this->configFactory->get('posthog_commerce.settings')->get('track_commerce_order_paid');
    if (!$orderPaidEnabled) {
      return;
    }
    $order = $event->getOrder();
    if (!empty($order)) {
      // @todo Implement this.
    }
  }

  /**
   * Listens and emits the posthog order completed event.
   *
   * @param \Drupal\state_machine\Event\WorkflowTransitionEvent $event
   *   The event.
   */
  public function orderCompleted(WorkflowTransitionEvent $event) {
    // Check, if the event should be tracked:
    $orderCompletedEnabled = $this->configFactory->get('posthog_commerce.settings')->get('track_commerce_order_completed');
    if (!$orderCompletedEnabled) {
      return;
    }
    try {
      /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
      $order = $event->getEntity();
      if (!empty($order)) {
        $customer = $order->getCustomer();
        $properties = [
          'checkout_id' => $order->id(),
          'order_id' => $order->getOrderNumber(),
          'total' => $order->getTotalPrice()->getNumber(),
          'currency' => $order->getTotalPrice()->getCurrencyCode(),
          'coupon' => !empty($order->collectAdjustments(['promotion'])) ? $order->collectAdjustments(['promotion'])[0]->getSourceId() : NULL,
        // @todo Get the products from the order:
          // 'products' => [],
        ];
        $this->posthogPhpSdk->captureUserEvent(self::POSTHOG_ORDER_COMPLETED_EVENT, $properties, [
          'commerce_order' => $order,
          'user' => $customer,
        ]);
      }
    }
    catch (\Exception $e) {
      if ($this->configFactory->get('posthog_php.settings')->get('enable_logging')) {
        $logger = $this->loggerChannelFactory->get('posthog_commerce');
        $logger->notice('Something went wrong, while capturing the posthog @event. Error message: @error_message', [
          '@event' => self::POSTHOG_ORDER_COMPLETED_EVENT,
          '@error_message' => $e->getMessage(),
        ]);
      }
    }

  }

}
