<?php

declare(strict_types=1);

namespace Drupal\posthog_php;

/**
 * Mirror of the PostHog PHP SDK.
 *
 * Mirrors https://github.com/PostHog/posthog-php/blob/master/lib/PostHog.php
 * as interface.
 *
 * All calls are initialized.
 *
 * Last updated on version 3.0.3.
 *
 * @see https://posthog.com/docs/libraries/php
 */
interface SdkDecoratorInterface {

  const CAPTURE_EVENT_PAGEVIEW = '$pageview';

  /**
   * Returns the current PostHog SDK version.
   *
   * E.g. 3.0.3
   *
   * @return string
   *   The version.
   */
  public static function getVersion(): string;

  /**
   * Initializes the default client to use.
   *
   * @throws Exception
   */
  public function init(): void;

  /**
   * Captures a user action.
   *
   * Expects a message as array, for example:
   * [
   *  'distinctId' => 'distinct_id_of_the_user',
   *   'event' => 'user_signed_up'
   * ]
   *
   * @param array $message
   *   The message to capture.
   * @param array $context
   *   Optional additional context array.
   *
   * @return bool
   *   Whether the capture call succeeded
   *
   * @throws Exception.
   *
   * @see https://posthog.com/docs/libraries/php#capturing-events
   */
  public function capture(array $message, array $context = []): bool;

  /**
   * Helper to capture an event on top of ::capture().
   *
   * @param string $event
   *   The event id.
   * @param array $properties
   *   Optional additional properties array, e.g.
   *   [
   *     'login_type' => 'email',
   *     'is_free_trial' => 'true',
   *   ]
   *     or
   *   [
   *     '$current_url' => 'https://example.com'
   *   ].
   * @param array $context
   *   Optional additional context array.
   *
   * @return bool
   *   Whether the capture call succeeded
   *
   * @see https://posthog.com/docs/libraries/php#capturing-events
   */
  public function captureUserEvent(string $event, array $properties = [], array $context = []): bool;

  /**
   * Helper to capture page view events on top of ::capture().
   *
   * @param string $currentUrl
   *   The current URL.
   * @param array $context
   *   Optional additional context array.
   *
   * @return bool
   *   Whether the capture call succeeded
   */
  public function captureUserPageView(string $currentUrl, array $context = []): bool;

  /**
   * Tags properties about the user.
   *
   * @param array $properties
   *   An associative array of user properties.
   *
   * @return bool
   *   Whether the identify call succeeded.
   *
   * @throws Exception.
   */
  public function identify(array $properties): bool;

  /**
   * Adds properties to a group.
   *
   * @param array $groupType
   *   The group type.
   * @param mixed $groupKey
   *   The group key.
   * @param array $properties
   *   An associative array of group properties.
   *
   * @return bool
   *   Whether the groupIdentify call succeeded.
   *
   * @throws Exception.
   */
  public function groupIdentify(string $groupType, mixed $groupKey, array $properties): bool;

  /**
   * Tells if the feature flag is enabled for this distinct id.
   *
   * @param string $key
   *   The feature flag key.
   * @param string $distinctId
   *   The distinct user id.
   * @param array $groups
   *   Optional additional groups array.
   * @param array $personProperties
   *   Optional additional person properties array.
   * @param array $groupProperties
   *   Optional additional group properties array.
   * @param bool $onlyEvaluateLocally
   *   Whether to only evaluate locally.
   * @param bool $sendFeatureFlagEvents
   *   Whether to send feature flag events.
   *
   * @return bool|null
   *   Whether the feature flag is enabled or null if not evaluated.
   *
   * @throws \Exception
   *   If an error occurs.
   */
  public function isFeatureEnabled(
    string $key,
    string $distinctId,
    array $groups = [],
    array $personProperties = [],
    array $groupProperties = [],
    bool $onlyEvaluateLocally = FALSE,
    bool $sendFeatureFlagEvents = TRUE,
  ): NULL | bool;

  /**
   * Retrieves the feature flag value for a given distinct ID.
   *
   * @param string $key
   *   The key of the feature flag.
   * @param string $distinctId
   *   The distinct ID to check the feature flag for.
   * @param array $groups
   *   (optional) An array of groups to consider.
   * @param array $personProperties
   *   (optional) An array of person properties to consider.
   * @param array $groupProperties
   *   (optional) An array of group properties to consider.
   * @param bool $onlyEvaluateLocally
   *   (optional) Whether to only evaluate the feature flag locally.
   * @param bool $sendFeatureFlagEvents
   *   (optional) Whether to send feature flag events.
   *
   * @return bool|string
   *   The value of the feature flag, which can be a boolean or a string.
   *
   * @throws \Exception
   *   Thrown when there is an error retrieving the feature flag value.
   */
  public function getFeatureFlag(
    string $key,
    string $distinctId,
    array $groups = [],
    array $personProperties = [],
    array $groupProperties = [],
    bool $onlyEvaluateLocally = FALSE,
    bool $sendFeatureFlagEvents = TRUE,
  ): null | bool | string;

  /**
   * Fetches the feature variants for a given distinct ID.
   *
   * @param string $distinctId
   *   The distinct user ID.
   * @param array $groups
   *   (optional) An array of groups to consider.
   *
   * @return array
   *   An array of feature variants.
   *
   * @throws \Exception
   *   Thrown when there is an error fetching the feature variants.
   */
  public function fetchFeatureVariants(string $distinctId, array $groups = []): array;

  /**
   * Aliases the distinct id.
   *
   * The alias can then be used as an alternative distinct id.
   *
   * @param mixed $distinctId
   *   The distinct id.
   * @param mixed $alias
   *   The alias.
   *
   * @return bool
   *   Whether the alias call succeeded.
   *
   * @throws Exception.
   */
  public function alias(mixed $distinctId, mixed $alias): bool;

  /**
   * Send a raw (prepared) message.
   *
   * @param array $message
   *   The raw message.
   *
   * @return bool
   *   Whether the alias call succeeded.
   */
  public function raw(array $message): bool;

  /**
   * Validate common properties.
   *
   * @param array $msg
   *   The message to validate.
   * @param string $type
   *   The type of validation.
   *
   * @throws \Exception
   *   If validation fails.
   */
  public function validate($msg, $type): void;

  /**
   * Flush the client.
   */
  public function flush(): bool;

}
