<?php

declare(strict_types=1);

namespace Drupal\posthog_php_events\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Posthog settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'posthog_php_events_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['posthog_php_events.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('posthog_php_events.settings');

    $form['hook_events'] = [
      '#type' => 'details',
      '#title' => $this->t('Hook events'),
      '#open' => TRUE,
    ];

    $form['hook_events']['track_drupal_user_login'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Track Drupal user login'),
      '#description' => $this->t('Event name is "User login".'),
      '#default_value' => $config->get('track_drupal_user_login'),
    ];
    $form['hook_events']['track_drupal_user_first_time_login'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Track Drupal user first time login (Mail verification)'),
      '#description' => $this->t('This event fires, after a newly registered user logs in the first time (= verifying the account, if "Require email verification when a visitor creates an account" is enabled). Event name is "User first time login".'),
      '#default_value' => $config->get('track_drupal_user_first_time_login'),
    ];
    $form['hook_events']['track_drupal_user_logout'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Track Drupal user logout'),
      '#description' => $this->t('Event name is "User logout".'),
      '#default_value' => $config->get('track_drupal_user_logout'),
    ];
    $form['hook_events']['track_drupal_user_registration'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Track Drupal user registration'),
      '#description' => $this->t("Tracks user registration made by the user. Note, that at this point he hasn't verified his account yet, if 'Require email verification when a visitor creates an account' is enabled). Event name is 'User registration'."),
      '#default_value' => $config->get('track_drupal_user_registration'),
    ];
    $form['hook_events']['track_drupal_user_block'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Track Drupal user block'),
      '#description' => $this->t('Event name is "User blocked".'),
      '#default_value' => $config->get('track_drupal_user_block'),
    ];
    $form['hook_events']['track_drupal_user_unblock'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Track Drupal user unblock'),
      '#description' => $this->t('Event name is "User unblocked".'),
      '#default_value' => $config->get('track_drupal_user_unblock'),
    ];
    $form['hook_events']['track_drupal_user_delete'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Track Drupal user deletion'),
      '#description' => $this->t('Event name is "User deleted".'),
      '#default_value' => $config->get('track_drupal_user_delete'),
    ];
    // @codingStandardsIgnoreStart
    // @todo Implement subscriber events:
    // $form['subscriber_events'] = [
    //   '#type' => 'details',
    //   '#title' => $this->t('Subscriber events'),
    //   '#open' => TRUE,
    // ];
    // @codingStandardsIgnoreEnd
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('posthog_php_events.settings')
      ->set('track_drupal_user_login', $form_state->getValue('track_drupal_user_login'))
      ->set('track_drupal_user_first_time_login', $form_state->getValue('track_drupal_user_first_time_login'))
      ->set('track_drupal_user_logout', $form_state->getValue('track_drupal_user_logout'))
      ->set('track_drupal_user_block', $form_state->getValue('track_drupal_user_block'))
      ->set('track_drupal_user_unblock', $form_state->getValue('track_drupal_user_unblock'))
      ->set('track_drupal_user_delete', $form_state->getValue('track_drupal_user_delete'))
      ->set('track_drupal_user_registration', $form_state->getValue('track_drupal_user_registration'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
