<?php

namespace Drupal\Tests\posthog_cookies\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\cookies\Traits\CookiesCacheClearTrait;

/**
 * Tests posthog_cookies Javascript related functionalities.
 *
 * @group posthog_cookies
 */
class PosthogCookiesFunctionalJavascriptTest extends WebDriverTestBase {
  use CookiesCacheClearTrait;

  /**
   * The user.
   *
   * @var \Drupal\user\Entity\User
   */
  protected $user;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'test_page_test',
    'filter_test',
    'cookies',
    'posthog_js',
    'posthog_cookies',
  ];

  /**
   * {@inheritDoc}
   */
  public function setUp(): void {
    parent::setUp();

    $this->config('system.site')->set('page.front', '/test-page')->save();
    $this->user = $this->drupalCreateUser([]);
    $this->drupalPlaceBlock('cookies_ui_block');
    // Set google_analytics settings:
    $this->config('posthog.settings')
      ->set('host', 'https://eu.i.posthog.com')
      ->set('api_key', 'phc_XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX')
      ->save();
    $this->config('posthog_js.settings')
      ->set('enabled', TRUE)
      ->set('cdn', TRUE)
      ->save();
    $this->clearBackendCaches();
  }

  /**
   * Tests if the cookies ga javascript file is correctly knocked in / out.
   */
  public function testPosthogCorrectlyKnocked() {
    $session = $this->assertSession();
    $driver = $this->getSession()->getDriver();

    $this->drupalGet('<front>');

    // Check that all required posthog javascript files are present:
    $session->elementsCount('css', 'script[src*="js/posthog_cookies.js"]', 1);
    $session->elementsCount('css', 'script[src*="https://eu-assets.i.posthog.com/array/phc_XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX"]', 1);
    $session->elementsCount('css', 'script[src*="posthog_js/js/cdn.js"]', 1);
    $session->elementsCount('css', 'script[src*="posthog_js/js/init.js"]', 1);

    // Check, that the posthog cookie is not set:
    $this->assertEmpty($driver->getCookie('ph_phc_XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX_posthog'));

    // Check, that the persistence is set to "memory" before accepting the
    // cookies:
    $persistence = $driver->evaluateScript("window.posthog.has_opted_out_capturing()");
    $this->assertEquals(TRUE, $persistence);

    // Fire consent script, accept all cookies:
    $consentScript = "var options = { all: true };
        document.dispatchEvent(new CustomEvent('cookiesjsrSetService', { detail: options }));";
    $driver->executeScript($consentScript);

    $this->drupalGet('<front>');

    // Check that all required posthog javascript files are still present:
    $session->elementsCount('css', 'script[src*="js/posthog_cookies.js"]', 1);
    $session->elementsCount('css', 'script[src*="https://eu-assets.i.posthog.com/array/phc_XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX"]', 1);
    $session->elementsCount('css', 'script[src*="posthog_js/js/cdn.js"]', 1);
    $session->elementsCount('css', 'script[src*="posthog_js/js/init.js"]', 1);

    // Check, that the posthog cookie is set now:
    $this->assertNotEmpty($driver->getCookie('ph_phc_XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX_posthog'));

    // Now, the persistence should be set to "localStorage+cookie":
    $persistence = $driver->evaluateScript("window.posthog.has_opted_out_capturing()");
    $this->assertEquals(FALSE, $persistence);

    // Now, remove the consent again:
    $removeConsentScript = "var options = { all: false };
    document.dispatchEvent(new CustomEvent('cookiesjsrSetService', { detail: options }));";
    $driver->executeScript($removeConsentScript);

    $this->drupalGet('<front>');

    // Check, that the posthog cookie is removed:
    $this->assertEmpty($driver->getCookie('ph_phc_XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX_posthog'));

    // Check, that the persistence is set to "memory"again:
    $persistence = $driver->evaluateScript("window.posthog.has_opted_out_capturing()");
    $this->assertEquals(TRUE, $persistence);
  }

  /**
   * Tests if the js file is correctly knocked in / out with js aggregation on.
   *
   * @todo Check, why this test currently fails, enabling js aggregation does
   * not work. Makes the test fail, but in reality they shouldn't. These infos
   * might help:
   * When changing the js aggregation, the following text will appear:
   * "These values are overridden. Changes on this form will be saved, but
   * overrides will take precedence. So maybe
   * https://www.drupal.org/docs/drupal-apis/configuration-api/configuration-override-system
   * might help to fix the problem? No idea, could also be some module missing
   * or we need to have a different installation profile.
   */
  public function testPosthogCorrectlyKnockedWithJsAggregation() {
    $this->drupalLogin($this->rootUser);
    $session = $this->assertSession();
    $page = $this->getSession()->getPage();
    $driver = $this->getSession()->getDriver();

    // Enable js aggregation:
    $this->drupalGet('/admin/config/development/performance');
    $page->checkField('edit-preprocess-js');
    $page->pressButton('edit-submit');
    $session->pageTextContains('The configuration options have been saved.');
    drupal_flush_all_caches();

    $this->drupalGet('<front>');

    // Check that the EU-Asset script is present, all others should get
    // aggregated:
    $session->elementsCount('css', 'script[src*="https://eu-assets.i.posthog.com/array/phc_XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX"]', 1);

    // Check, that the posthog cookie is not set:
    $this->assertEmpty($driver->getCookie('ph_phc_XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX_posthog'));

    // Check, that the persistence is set to "memory" before accepting the
    // cookies:
    $persistence = $driver->evaluateScript("window.posthog.has_opted_out_capturing()");
    $this->assertEquals(TRUE, $persistence);

    // Fire consent script, accept all cookies:
    $consentScript = "var options = { all: true };
        document.dispatchEvent(new CustomEvent('cookiesjsrSetService', { detail: options }));";
    $driver->executeScript($consentScript);

    $this->drupalGet('<front>');

    // Check that the EU-Asset script is still present, all others should get
    // aggregated:
    $session->elementsCount('css', 'script[src*="https://eu-assets.i.posthog.com/array/phc_XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX"]', 1);

    // Check, that the posthog cookie is set now:
    $this->assertNotEmpty($driver->getCookie('ph_phc_XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX_posthog'));

    // Now, the persistence should be set to "localStorage+cookie":
    $persistence = $driver->evaluateScript("window.posthog.has_opted_out_capturing()");
    $this->assertEquals(FALSE, $persistence);

    // Now, remove the consent again:
    $removeConsentScript = "var options = { all: false };
    document.dispatchEvent(new CustomEvent('cookiesjsrSetService', { detail: options }));";
    $driver->executeScript($removeConsentScript);

    $this->drupalGet('<front>');

    // Check, that the posthog cookie is removed:
    $this->assertEmpty($driver->getCookie('ph_phc_XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX_posthog'));

    // Check, that the persistence is set to "memory"again:
    $persistence = $driver->evaluateScript("window.posthog.has_opted_out_capturing()");
    $this->assertEquals(TRUE, $persistence);
  }

  /**
   * Tests, that the session id is reset after every call.
   *
   * Except when we change the persistence (allow the cookies).
   *
   * @see https://www.drupal.org/project/posthog/issues/3506122
   */
  public function testSessionIdResetPersistenceMemory() {
    $driver = $this->getSession()->getDriver();

    // Check, that the posthog cookie is not set:
    $this->assertEmpty($driver->getCookie('ph_phc_XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX_posthog'));

    // Go to the first page:
    $this->drupalGet('/test-page');
    // Create the session id script and get the session id:
    $sessionIdScript = "window.posthog.get_session_id()";
    $originalSessionId = $driver->evaluateScript($sessionIdScript);

    // There should be no session id, since we opt out:
    $this->drupalGet('/test-render-title');
    $sessionId = $driver->evaluateScript($sessionIdScript);
    $this->assertEmpty($sessionId);

    $this->drupalGet('/test-page-static-title');
    $sessionId = $driver->evaluateScript($sessionIdScript);
    $this->assertEmpty($sessionId);

    // Fire COOKiES consent script, accept all cookies:
    $consentScript = "var options = { all: true };
        document.dispatchEvent(new CustomEvent('cookiesjsrSetService', { detail: options }));";
    $driver->executeScript($consentScript);

    $this->drupalGet('<front>');

    // Create the session id script and get the session id:
    $sessionIdScript = "window.posthog.get_session_id()";
    $originalSessionId = $driver->evaluateScript($sessionIdScript);

    // The session id should be the same on every page now:
    $this->drupalGet('/test-render-title');
    $sessionId = $driver->evaluateScript($sessionIdScript);
    $this->assertEquals($originalSessionId, $sessionId);

    $this->drupalGet('/test-page-static-title');
    $sessionId = $driver->evaluateScript($sessionIdScript);
    $this->assertEquals($originalSessionId, $sessionId);

    // Check, that the posthog cookie is set now:
    $this->assertNotEmpty($driver->getCookie('ph_phc_XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX_posthog'));
  }

}
