<?php

namespace Drupal\posthog_eca\Plugin\Action;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Sends a warning message to the current user's screen.
 *
 * @Action(
 *   id = "eca_posthog_event",
 *   label = @Translation("Emits a configurable posthog event"),
 *   type = "system"
 * )
 */
class PosthogEvent extends ConfigurableActionBase {

  /**
   * Renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected $token;

  /**
   * PostHog SDK service.
   *
   * @var \Drupal\posthog\PosthogSdk
   */
  protected $posthogSdk;

  /**
   * User attributes provider service.
   *
   * @var \Drupal\posthog\UserAttributesProvider
   */
  protected $userAttributesProvider;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->renderer = $container->get('renderer');
    $instance->token = $container->get('token');
    $instance->posthogSdk = $container->get('posthog_php.sdk_decorator');
    $instance->userAttributesProvider = $container->get('posthog.user_attributes_provider');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'event_name' => '',
      'custom' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['event_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Posthog event name'),
      '#description' => $this->t("Enter the name of the posthog event to fire, when the action is executed."),
      '#default_value' => $this->configuration['event_name'],
      '#required' => TRUE,
    ];
    $form['custom'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom properties (YAML)'),
      '#description' => $this->t('Enter the setting name and value as <a href=":link" target="_blank">YAML</a>.', [
        ':link' => 'https://quickref.me/yaml.html',
      ]),
      '#default_value' => $this->configuration['custom'],
    ];
    $token_tree = [
      '#theme' => 'token_tree_link',
      '#token_types' => [],
    ];
    try {
      $rendered_token_tree = $this->renderer->render($token_tree);
    }
    catch (\Exception $e) {
      $rendered_token_tree = '';
    }
    $form['custom']['#description'] = ' ' . $this->t('This field supports tokens. @browse_tokens_link', [
      '@browse_tokens_link' => $rendered_token_tree,
    ]);
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::validateConfigurationForm($form, $form_state);
    $customValue = $form_state->getValue('custom');
    if (!empty($customValue)) {
      $parsedCustomValue = $this->token->replace($customValue);
      try {
        Yaml::decode($parsedCustomValue);
      }
      catch (\Exception $e) {
        $form_state->setError($form['custom'], $this->t('Invalid YAML code.'));
      }
    }

  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['event_name'] = $form_state->getValue('event_name');
    $this->configuration['custom'] = $form_state->getValue('custom');
    parent::submitConfigurationForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   *
   * Mainly copied from parent execute method, except for a different messenger
   * instruction.
   */
  public function execute(mixed $entity = NULL): void {
    $properties = [];
    if (!empty($this->configuration['custom'])) {
      $customPropertiesString = $this->configuration['custom'];
      $parsedCustomPropertiesString = $this->token->replace($customPropertiesString);
      $yaml = Yaml::decode($parsedCustomPropertiesString);
      $properties = array_merge($properties, $yaml);
    }
    $this->posthogSdk->captureUserEvent(
      $this->configuration['event_name'],
      $properties,
    );
  }

}
