<?php

namespace Drupal\posthog_webform\Plugin\WebformHandler;

use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\Core\Utility\Token;
use Drupal\posthog\UserAttributesProvider;
use Drupal\posthog_php\SdkDecoratorInterface;
use Drupal\webform\Plugin\WebformHandlerBase;
use Drupal\webform\WebformSubmissionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\posthog_php\MissingPosthogApiKeyException;

/**
 * Posthog webform handler.
 *
 * @WebformHandler(
 *   id = "posthog_webform",
 *   label = @Translation("Posthog event"),
 *   category = @Translation("External"),
 *   description = @Translation("Emits a customizable posthog event (server-side) on form submission."),
 *   cardinality = \Drupal\webform\Plugin\WebformHandlerInterface::CARDINALITY_UNLIMITED,
 *   results = \Drupal\webform\Plugin\WebformHandlerInterface::RESULTS_PROCESSED,
 *   submission = \Drupal\webform\Plugin\WebformHandlerInterface::SUBMISSION_REQUIRED,
 * )
 */
class PosthogWebformHandler extends WebformHandlerBase {

  use StringTranslationTrait;

  /**
   * The user attributes provider.
   *
   * @var \Drupal\posthog\UserAttributesProvider
   */
  protected UserAttributesProvider $userAttributesProvider;

  /**
   * The Posthog PHP SDK.
   *
   * @var \Drupal\posthog_php\SdkDecoratorInterface
   */
  protected ?SdkDecoratorInterface $posthogSdk;

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected Token $token;

  /**
   * The logger channel factory service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected LoggerChannelFactoryInterface $loggerChannelFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->userAttributesProvider = $container->get('posthog.user_attributes_provider');
    try {
      $instance->posthogSdk = $container->get('posthog_php.sdk_decorator');
    }
    catch (MissingPosthogApiKeyException) {
      $instance->posthogSdk = NULL;
    }
    $instance->token = $container->get('token');
    $instance->loggerChannelFactory = $container->get('logger.factory');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'event_name' => 'webform_submission',
      'send_current_url' => FALSE,
      'send_submission_id' => FALSE,
      'custom' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form['settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Posthog event settings'),
    ];
    $form['settings']['event_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Posthog event name (post-save)'),
      '#description' => $this->t("Enter the name of the posthog event to fire, when the webform is submitted (Default: 'webform_submission')."),
      '#default_value' => $this->configuration['event_name'],
      '#required' => TRUE,
    ];
    $form['settings']['send_current_url'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Send current URL'),
      '#description' => $this->t('Sends the current URL as an event property.'),
      '#default_value' => $this->configuration['send_current_url'],
    ];
    $form['settings']['send_submission_id'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Send submission ID'),
      '#description' => $this->t('Sends the webform submission ID as an event property.'),
      '#default_value' => $this->configuration['send_submission_id'],
    ];
    $form['settings']['custom'] = [
      '#type' => 'webform_codemirror',
      '#mode' => 'yaml',
      '#title' => $this->t('Custom properties (YAML)'),
      '#placeholder' => 'property_name: property_value',
      '#description' => $this->t('Enter the setting name and value as <a href=":link" target="_blank">YAML</a>.', [
        ':link' => 'https://quickref.me/yaml.html',
      ]),
      '#default_value' => $this->configuration['custom'],
      // Must set #parents.
      // @see \Drupal\webform\Plugin\WebformHandler\SettingsWebformHandler::submitConfigurationForm
      '#parents' => ['settings', 'custom'],
    ];
    $token_tree = [
      '#theme' => 'token_tree_link',
      '#token_types' => [],
    ];
    try {
      $rendered_token_tree = $this->renderer->render($token_tree);
    }
    catch (\Exception $e) {
      $rendered_token_tree = '';
    }
    $form['settings']['custom']['#description'] = ' ' . $this->t('This field supports tokens. @browse_tokens_link', [
      '@browse_tokens_link' => $rendered_token_tree,
    ]);
    return $this->setSettingsParents($form);
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    $this->configuration['event_name'] = $form_state->getValue('event_name');
    $this->configuration['send_current_url'] = $form_state->getValue('send_current_url');
    $this->configuration['send_submission_id'] = $form_state->getValue('send_submission_id');
    $this->configuration['custom'] = $form_state->getValue('custom');
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(WebformSubmissionInterface $webform_submission, $update = TRUE) {
    if ($this->posthogSdk == NULL) {
      $this->loggerChannelFactory->get('posthog_webform')->log('warning', 'Could not capture PostHog user event, because the PostHog API key is missing.');
      return;
    }
    $properties = [];
    // Add the current URL to the event properties, if enabled:
    if ($this->configuration['send_current_url']) {
      $properties['$current_url'] = Url::fromRoute('<current>', [], ['absolute' => 'true'])->toString();
    }
    // Add the webform submission id, if enabled:
    if ($this->configuration['send_submission_id']) {
      $properties['webform_submission_id'] = $webform_submission->id();
    }
    if (!empty($this->configuration['custom'])) {
      $customPropertiesString = $this->configuration['custom'];
      $parsedCustomPropertiesString = $this->token->replace($customPropertiesString);
      $yaml = Yaml::decode($parsedCustomPropertiesString);
      $properties = array_merge($properties, $yaml);
    }
    // Always add the "$event_type" "submit" property:
    $properties['$event_type'] = 'submit';
    $this->posthogSdk->captureUserEvent(
      $this->configuration['event_name'],
      $properties,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getSummary() {
    return [
      '#type' => 'markup',
      '#markup' => $this->t('<strong>Event name:</strong> @event_name<br><strong>Send current URL:</strong> @send_current_url<br><strong>Send submission ID:</strong> @send_submission_id', [
        '@event_name' => $this->configuration['event_name'],
        '@send_current_url' => $this->configuration['send_current_url'] ? $this->t('Yes') : $this->t('No'),
        '@send_submission_id' => $this->configuration['send_submission_id'] ? $this->t('Yes') : $this->t('No'),
      ]),
    ];
  }

}
