<?php

declare(strict_types=1);

namespace Drupal\posthog\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Posthog settings for this site.
 */
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Cache\CacheBackendInterface;

/**
 * The posthog settings form.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * Constructs a SettingsForm object.
   *
   * @param \Drupal\Core\Cache\CacheBackendInterface $cacheBackend
   *   The cache backend.
   */
  public function __construct(
    protected CacheBackendInterface $cacheBackend,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('cache.discovery'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'posthog_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['posthog.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('posthog.settings');

    $form['general_configuration'] = [
      '#type' => 'details',
      '#title' => $this->t('General configuration'),
      '#open' => TRUE,
    ];
    $form['general_configuration']['host'] = [
      '#type' => 'url',
      '#title' => $this->t('Posthog host'),
      '#required' => TRUE,
      '#default_value' => $config->get('host'),
      '#default' => $this->t("The Posthog host to communicate with, e.g. https://eu.i.posthog.com or https://us.i.posthog.com or your custom Posthog host. See https://posthog.com/docs/api for details."),
    ];
    $form['general_configuration']['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Posthog project API key'),
      '#required' => TRUE,
      '#default_value' => $config->get('api_key'),
      '#default' => $this->t('Your posthog project API key. You can find your project API key and instance address in the <a rel="noopener noreferrer" href="https://app.posthog.com/project/settings" target="_blank">project settings</a> page in PostHog.'),
    ];
    $form['user_identifying'] = [
      '#type' => 'details',
      '#title' => $this->t('Enhanced user identification'),
      '#open' => TRUE,
    ];
    $form['user_identifying']['identify_anonymous'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Identify anonymous users'),
      '#default_value' => $config->get('identify_anonymous'),
      '#description' => $this->t('Identify anonymous users, meaning person profiles will be created for them.<br><strong>Note:</strong> Enabling this setting will make all events identified events. Identified events can cost more than anonymous events. See <a href=:link1>Anonymous vs identified events</a> for more information.', [
        ':link1' => 'https://posthog.com/docs/data/anonymous-vs-identified-events',
      ]),
    ];
    $form['user_identifying']['distinct_id'] = [
      '#type' => 'radios',
      '#title' => $this->t('Identify logged in users'),
      '#options' => [
        'no_identification' => $this->t('No enhanced identification (Posthog default)'),
        'uid' => $this->t('Identify by Drupal user ID'),
        'mail' => $this->t('Identify by Drupal user email'),
        'name' => $this->t('Identify by Drupal user name'),
      ],
      '#default_value' => $config->get('distinct_id'),
      '#description' => $this->t('Select which unique identifier should be used as the PostHog "distinct_id" to identify user events.<br><br><strong>NOTE:</strong> Setting this to anything other than "No enhanced identification (Posthog default)" will make all events from logged in users identified events. Identified events can cost more than anonymous events. See <a href=:link1>Anonymous vs identified events</a> for more information.', [
        ':link1' => 'https://posthog.com/docs/data/anonymous-vs-identified-events',
      ]),
    ];
    $form['user_identifying']['user_properties'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('User properties'),
      '#default_value' => $config->get('user_properties'),
      '#states' => [
        'invisible' => [
          ':input[name="distinct_id"]' => ['value' => 'no_identification'],
        ],
      ],
      '#options' => [
        'email' => $this->t('Drupal user email'),
        'name' => $this->t('Drupal user name'),
        'uid' => $this->t('Drupal user ID'),
      ],
      '#description' => $this->t('Define additional user properties to identify the user with (e.g. email, name, etc.).<br><br>Note, that these properties will be used by Posthog internally, to display the user in the Posthog interface. The following hierarchy is used: <strong>Mail > Name > ID</strong>. If no user property is selected, Posthog will use the "distinct_id" (see "Identify logged in users" above) to display the user.'),
      '#open' => TRUE,
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('posthog.settings')
      ->set('host', $form_state->getValue('host'))
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('distinct_id', $form_state->getValue('distinct_id'))
      ->set('user_properties', $form_state->getValue('user_properties'))
      ->set('identify_anonymous', $form_state->getValue('identify_anonymous'))
      ->save();
    // Clear the discovery cache to make sure the new settings are applied:
    $this->cacheBackend->deleteAll();
    parent::submitForm($form, $form_state);
  }

}
