<?php

declare(strict_types=1);

namespace Drupal\posthog;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Session\AccountInterface;

/**
 * Provides posthog user properties.
 */
final class UserAttributesProvider {

  /**
   * The posthog settings configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $posthogSettings;

  /**
   * Constructs a PropertyProvider object.
   */
  public function __construct(
    private readonly ConfigFactoryInterface $configFactory,
    private readonly AccountInterface $currentUser,
    private readonly CookieReader $cookieReader,
  ) {
    $this->posthogSettings = $this->configFactory->get('posthog.settings');
  }

  /**
   * Checks if the user should be identified.
   *
   * Note, that server side, it only makes sense to identify anonymous users,
   * if the user has given consent (a cookie is set).
   */
  public function shouldUserBeIdentified(?AccountInterface $user = NULL): ?bool {
    return $this->getUserDistinctIdBySetting($user) !== NULL ||
      ($this->posthogSettings->get('identify_anonymous') && $this->cookieReader->getDistinctIdByCookie() !== NULL);
  }

  /**
   * Returns a user's distinct id based on the configuration.
   *
   * Note, usually it is wiser to use "getCurrentUserDistinctId" instead.
   */
  public function getUserDistinctIdBySetting(?AccountInterface $user = NULL): ?string {
    if ($user === NULL) {
      $user = $this->currentUser;
    }
    // Anonymous users, can not be identified by their user id, email or name,
    // so we simply return NULL:
    if ($user->isAnonymous()) {
      return NULL;
    }
    switch ($this->posthogSettings->get('distinct_id')) {
      case 'uid':
        return (string) $user->id();

      case 'mail':
        return $user->getEmail();

      case 'name':
        return $user->getAccountName();

      default:
      case 'no_identification':
        return NULL;
    }
    return NULL;
  }

  /**
   * Returns the current user's distinct id based on the configuration.
   */
  public function getCurrentUserDistinctId(): ?string {
    return $this->cookieReader->getDistinctIdByCookie() ?? $this->getUserDistinctIdBySetting();
  }

  /**
   * Retrieves user properties.
   *
   * @return array
   *   An associative array of user properties.
   */
  public function getUserProperties(AccountInterface $user): array {
    $userProperties = [];

    // Get the user properties from the configuration and set them inside the
    // user properties array:
    $userSettingsProperties = $this->posthogSettings->get('user_properties');

    if (!empty($userSettingsProperties['uid'])) {
      $userProperties['uid'] = $user->id();
    }
    if (!empty($userSettingsProperties['email'])) {
      $userProperties['email'] = $user->getEmail();
    }
    if (!empty($userSettingsProperties['name'])) {
      $userProperties['name'] = $user->getAccountName();
    }

    return $userProperties;
  }

  /**
   * Retrieves current user properties.
   */
  public function getCurrentUserProperties(): array {
    return $this->getUserProperties($this->currentUser);
  }

}
