<?php

declare(strict_types=1);

namespace Drupal\posthog_js\Form;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\posthog_js\ConditionsFormTrait;

/**
 * Configure Tab Title Attention settings for this site.
 */
final class SettingsForm extends ConfigFormBase {
  // @todo Use the core "ConditionsFormTrait" once that exists.
  // @see https://www.drupal.org/project/drupal/issues/3506721.
  use ConditionsFormTrait;

  /**
   * Constructs a \Drupal\posthog_js\Form\SettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cacheBackend
   *   The cache backend.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    protected CacheBackendInterface $cacheBackend,
    ?TypedConfigManagerInterface $typedConfigManager = NULL,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('config.factory'),
      $container->get('cache.discovery'),
      $container->get('config.typed'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'posthog_js_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['posthog_js.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('posthog_js.settings');

    $form['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#description' => $this->t("Check to add the JavaScript tracking code to matching pages."),
      '#default_value' => $config->get('enabled'),
    ];

    $form['cdn'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Load from Posthog ("CDN" [recommended])'),
      '#description' => $this->t("Check to load Posthog JavaScript from Posthog (recommended).<br />If disabled, you must provide the (posthog-js) library yourself, for example using <code>composer require npm-asset/posthog-js</code>.<br /><a href='https://posthog.com/docs/privacy/gdpr-compliance' target='_blank'></a>Note that Posthog points out its GDPR compliance.</a>"),
      '#default_value' => $config->get('cdn'),
      '#states' => [
        'invisible' => [
          ':input[name="enabled"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $accessJsSettings = $this->currentUser()->hasPermission('administer posthog_js settings javascript');
    $accessJsSettingsWarning = !$accessJsSettings ? ' <em>' . $this->t('This field has been disabled because you do not have the "administer posthog_js settings javascript" permission, which is required here.') . '</em>' : '';
    $form['init_config_json'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Initialization configuration (JSON)'),
      '#description' => $this->t('Optional additional Posthog initialization JSON code. See the <a href=":link1" target="_blank">help page</a> for further help on this. For a full list of all available options, see <a href=":link2" target="_blank">here</a>. Leave empty to use the defaults.', [
        ':link1' => '/admin/help/posthog',
        ':link2' => 'https://posthog.com/docs/configure/initialization-configuration',
      ]) . $accessJsSettingsWarning,
      '#default_value' => $config->get('init_config_json') ?? "{}",
      '#placeholder' => "{\n  // autocapture: true,\n  // capture_pageview: true,\n}",
      '#disabled' => !$accessJsSettings,
      '#states' => [
        'invisible' => [
          ':input[name="enabled"]' => ['checked' => FALSE],
        ],
      ],
    ];

    // Build the conditions form:
    $conditions = $this->config('posthog_js.settings')->get('conditions');

    $form[$this->conditionsFormKey] = $this->buildConditionsForm([], $form_state, 'posthog_js', $conditions);

    // Hide conditions, when the module is disabled:
    $form[$this->conditionsFormKey]['condition_tabs']['#states']['invisible'][':input[name="enabled"]'] = ['checked' => FALSE];

    $form['conditions_logic'] = [
      '#type' => 'radios',
      '#description' => $this->t('Choose whether all conditions must be met (AND) or any condition must be met (OR).'),
      '#options' => [
        'and' => $this->t('AND'),
        'or' => $this->t('OR'),
      ],
      '#default_value' => $this->config('posthog_js.settings')->get('conditions_logic') ?? 'and',
      '#states' => [
        'invisible' => [
          ':input[name="enabled"]' => ['checked' => FALSE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Retrieves filtered condition definitions for a given consumer.
   *
   * @param string $consumer
   *   A string identifying the consumer of the condition definitions (e.g.
   *   the module id).
   *
   * @return array
   *   An array of filtered condition definitions.
   */
  protected function getFilteredConditionDefinitions($consumer) {
    $filteredConditions = $this->getConditionManager()->getFilteredDefinitions(
      $consumer,
      $this->getContextRepository()->getAvailableContexts(),
    );
    // @todo We are currently not showing the current theme condition, as
    // there is currently no possibility, to select "any" theme:
    // @see https://www.drupal.org/project/drupal/issues/2783897
    unset($filteredConditions['current_theme']);
    return $filteredConditions;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    try {
      // Validate JSON:
      $initConfigJson = trim($form_state->getValue('init_config_json') ?? '');
      if (!empty($initConfigJson)) {
        $form_state->setValue('init_config_json', json_encode(json_decode($initConfigJson ?: '{}', FALSE, 5, JSON_THROW_ON_ERROR | JSON_INVALID_UTF8_SUBSTITUTE), JSON_FORCE_OBJECT | JSON_INVALID_UTF8_SUBSTITUTE | JSON_PRETTY_PRINT | JSON_THROW_ON_ERROR));
      }
    }
    catch (\JsonException $e) {
      $form_state->setErrorByName('init_config_json', $this->t('Invalid JSON'));
    }

    $this->validateConditionsForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('posthog_js.settings')
      ->set('enabled', $form_state->getValue('enabled'))
      ->set('cdn', $form_state->getValue('cdn'))
      ->set('init_config_json', $form_state->getValue('init_config_json'))
      ->set($this->conditionsFormKey, $this->submitConditionsForm($form, $form_state))
      ->set('conditions_logic', $form_state->getValue('conditions_logic'))
      ->save();
    // Clear the discovery cache to make sure the new settings are applied:
    // @todo Improve this! We should never simply delete all cache, but only the
    // cache for the posthog_js module:
    $this->cacheBackend->deleteAll();
    parent::submitForm($form, $form_state);
  }

}
