<?php

declare(strict_types=1);

namespace Drupal\posthog_php;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\posthog\UserAttributesProvider;
use PostHog\PostHog;

/**
 * Mirrors the PostHog class from PostHog SDK.
 */
class SdkDecorator implements SdkDecoratorInterface {

  /**
   * Constructs a SdkDecorator object.
   */
  public function __construct(
    readonly ConfigFactoryInterface $configFactory,
    readonly LoggerChannelFactoryInterface $loggerChannelFactory,
    readonly ModuleHandlerInterface $moduleHandler,
    readonly UserAttributesProvider $userAttributesProvider,
    protected $initialized = FALSE,
  ) {
    $this->init();
  }

  /**
   * {@inheritDoc}
   */
  public static function getVersion(): string {
    return PostHog::VERSION;
  }

  /**
   * {@inheritdoc}
   */
  public function init(): void {
    if ($this->initialized) {
      return;
    }
    $config = $this->configFactory->get('posthog.settings');
    $apiKey = $config->get('api_key');
    if ($apiKey == '' || $apiKey == NULL) {
      throw new MissingPosthogApiKeyException();
    }
    $host = $config->get('host');
    PostHog::init($apiKey,
      ['host' => $host]
    );
    $this->initialized = TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function identify(array $properties = []): bool {
    if (!$this->initialized) {
      $this->init();
    }
    // If the user should not be identified, return early:
    if (!$this->userAttributesProvider->shouldUserBeIdentified()) {
      return FALSE;
    }
    // Note, that the server side "identify" method requires a distinct id:
    $distinctId = $this->userAttributesProvider->getCurrentUserDistinctId();
    // If the distinct id is NULL (no cookie set), we can't and don't want to
    // identify the user. Return FALSE:
    if ($distinctId === NULL) {
      return FALSE;
    }
    $message = [
      'distinctId' => $distinctId,
    ];
    if (!empty($properties)) {
      $capture['properties'] = $properties;
    }

    $success = PostHog::identify($message);

    if (!$success && (bool) $this->configFactory->get('posthog.settings')->get('enable_logging')) {
      $this->loggerChannelFactory->get('posthog_php')->error('Failed to identify user with distinct id: %did.', [
        '%did' => $distinctId,
      ]);
    }
    return $success;
  }

  /**
   * {@inheritdoc}
   */
  public function groupIdentify(string $groupType, mixed $groupKey, array $properties = []): bool {
    if (!$this->initialized) {
      $this->init();
    }
    $message = [
      'groupType' => $groupType,
      'groupKey' => $groupKey,
    ];
    if (!empty($properties)) {
      $capture['properties'] = $properties;
    }

    $success = PostHog::groupIdentify($message);
    if (!$success && (bool) $this->configFactory->get('posthog.settings')->get('enable_logging')) {
      $this->loggerChannelFactory->get('posthog_php')->error('Failed to group identify group with group key: %gKey.', [
        '%gKey' => $groupKey,
      ]);
    }
    return $success;
  }

  /**
   * {@inheritdoc}
   */
  public function isFeatureEnabled(string $key, string $distinctId, array $groups = [], array $personProperties = [], array $groupProperties = [], bool $onlyEvaluateLocally = FALSE, bool $sendFeatureFlagEvents = TRUE): ?bool {
    if (!$this->initialized) {
      $this->init();
    }
    return PostHog::isFeatureEnabled($key, $distinctId, $groups, $personProperties, $groupProperties, $onlyEvaluateLocally, $sendFeatureFlagEvents);
  }

  /**
   * {@inheritdoc}
   */
  public function getFeatureFlag(string $key, string $distinctId, array $groups = [], array $personProperties = [], array $groupProperties = [], bool $onlyEvaluateLocally = FALSE, bool $sendFeatureFlagEvents = TRUE): null|bool|string {
    if (!$this->initialized) {
      $this->init();
    }
    return PostHog::getFeatureFlag($key, $distinctId, $groups, $personProperties, $groupProperties, $onlyEvaluateLocally, $sendFeatureFlagEvents);
  }

  /**
   * {@inheritdoc}
   */
  public function getAllFlags(string $distinctId, array $groups = [], array $personProperties = [], array $groupProperties = [], bool $onlyEvaluateLocally = FALSE): array {
    if (!$this->initialized) {
      $this->init();
    }
    return PostHog::getAllFlags($distinctId, $groups, $personProperties, $groupProperties, $onlyEvaluateLocally);
  }

  /**
   * {@inheritdoc}
   */
  public function fetchFeatureVariants(string $distinctId, array $groups = []): array {
    if (!$this->initialized) {
      $this->init();
    }
    return PostHog::fetchFeatureVariants($distinctId, $groups);
  }

  /**
   * {@inheritdoc}
   */
  public function alias(mixed $distinctId, mixed $alias): bool {
    if (!$this->initialized) {
      $this->init();
    }
    $message = [
      'distinctId' => $distinctId,
      'alias' => $alias,
    ];
    $success = PostHog::alias($message);

    if (!$success && (bool) $this->configFactory->get('posthog.settings')->get('enable_logging')) {
      $this->loggerChannelFactory->get('posthog_php')->error('Failed to alias user with distinct id: %did.', [
        '%did' => $distinctId,
      ]);
    }

    return $success;
  }

  /**
   * {@inheritdoc}
   */
  public function raw(array $message): bool {
    if (!$this->initialized) {
      $this->init();
    }
    return PostHog::raw($message);
  }

  /**
   * {@inheritdoc}
   */
  public function validate($msg, $type): void {
    if (!$this->initialized) {
      $this->init();
    }
    PostHog::validate($msg, $type);
  }

  /**
   * {@inheritdoc}
   */
  public function flush(): bool {
    if (!$this->initialized) {
      $this->init();
    }
    return PostHog::flush();
  }

  /**
   * {@inheritDoc}
   */
  public function capture(array $message, array $context = []): bool {
    if (!$this->initialized) {
      $this->init();
    }
    $config = $this->configFactory->get('posthog_php.settings');

    // If the distinct id is empty (e.g. no cookie set), we want to generate our
    // own id, since "distinctId" is required by PostHog:
    if (empty($message['distinctId'])) {
      $message['distinctId'] = uniqid('anonymous-');
    }

    // If the user should not be identified, we don't want to process person
    // profiles:
    if (!$this->userAttributesProvider->shouldUserBeIdentified()) {
      $message['properties']['$process_person_profile'] = FALSE;
    }

    $this->moduleHandler->alter('posthog_capture', $message, $context);
    $success = PostHog::capture($message);

    if (!$success && (bool) $config->get('enable_logging')) {
      $this->loggerChannelFactory->get('posthog_php')->error('Failed to capture user event for user with distinct id: %did.', [
        '%did' => $message['distinctId'],
      ]);
    }

    return $success;
  }

  /**
   * {@inheritDoc}
   */
  public function captureUserEvent(string $event, array $properties = [], array $context = []): bool {
    // Try to get the distinct id:
    $distinctId = $this->userAttributesProvider->getCurrentUserDistinctId();
    $capture = [
      'distinctId' => $distinctId,
      'event' => $event,
    ];

    if (!empty($properties)) {
      $capture['properties'] = $properties;
    }

    return $this->capture($capture, $context);
  }

  /**
   * {@inheritDoc}
   */
  public function captureUserPageView(string $currentUrl, array $context = []): bool {
    return $this->captureUserEvent(SdkDecoratorInterface::CAPTURE_EVENT_PAGEVIEW, ['$current_url' => $currentUrl], $context);
  }

}
